/*
 * Copyright (c) 2022-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#if defined(ARM_COMPUTE_ENABLE_SME2)

#include <algorithm>
#include <cstddef>

namespace arm_conv {
namespace depthwise {

void sme2_fp32_planar_5x5_s1_4rows_mla_za_impl(
  const float *inptr,
  size_t ld_in_row,
  size_t ld_in_col,
  size_t ld_in_vl,
  unsigned int pad_top,
  unsigned int valid_input_rows,
  unsigned int pad_left,
  unsigned int valid_input_cols,
  const float *weights,
  const float *bias,
  float **outptrs,
  const size_t *outlds,
  const size_t *outvllds,
  unsigned int output_cols,
  unsigned int start_channel,
  unsigned int valid_channels,
  float act_min,
  float act_max
)
{
  struct Args
  {
    const float *inptr;
    size_t ld_in_vl;
    long unsigned int pad_top, pad_bottom, pad_left;
    const float *weights;
    const float *bias;
    long unsigned int input_cols, output_cols;
    float **outptrs;
    const size_t *ld_out_cols;
    const size_t *ld_out_vls;
    long unsigned int current_channel, n_channels;
    float clamp_min, clamp_max;
  };

  Args args = { inptr, ld_in_vl, pad_top, 8u - std::min(8u, pad_top + valid_input_rows), pad_left, weights, bias, valid_input_cols, output_cols, outptrs, outlds, outvllds, start_channel, valid_channels, act_min, act_max };

  __asm__ __volatile__(
    "ldr x6, [%x[args], %[offsetof_Args_pad_bottom]]\n"
    "mov x20, #0x8\n"
    ".inst 0xd503477f  // SMSTART ZA\n"
    "ldr x7, [%x[args], %[offsetof_Args_pad_top]]\n"
    "ptrue p2.b\n"
    ".inst 0x25207812  // ptrue pn10.b\n"
    "ldr x17, [%x[args], %[offsetof_Args_n_channels]]\n"
    "ld1rw { z23.s }, p2/Z, [%x[args], %[offsetof_Args_clamp_min]]\n"
    "sub x20, x20, x6\n"
    "ldr x16, [%x[args], %[offsetof_Args_current_channel]]\n"
    "ld1rw { z29.s }, p2/Z, [%x[args], %[offsetof_Args_clamp_max]]\n"
    "whilelt p1.s, XZR, x17\n"
    "whilelt p9.s, XZR, x20\n"
    "whilelt p8.s, XZR, x7\n"
    "eor p8.b, p2/Z, p8.b, p9.b\n"
    "1:"  // Channel loop
    "ldr x20, [%x[args], %[offsetof_Args_bias]]\n"
    "fmov z24.s, #0x0\n"
    "cbz x20, 2f\n"
    "ld1w { z24.s }, p1/Z, [x20, x16, LSL #2]\n"
    "2:"  // Load bias: Done
    "ldr x15, [%x[args], %[offsetof_Args_input_cols]]\n"
    "mov x22, #0x8\n"
    "add x20, x7, x6\n"
    "mov z25.d, z24.d\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    "lsl x21, %x[ld_in_row], #0x2\n"
    "mov z26.d, z24.d\n"
    "mov z27.d, z24.d\n"
    "ldr x13, [%x[args], %[offsetof_Args_inptr]]\n"
    "mov x8, #0x0\n"
    "sub x22, x22, x20\n"
    "sub x20, x15, #0x1\n"
    "ldr x11, [%x[args], %[offsetof_Args_output_cols]]\n"
    "orr x20, x20, %x[ld_in_col], LSL #18\n"
    ".inst 0xa14049c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa14149c4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "orr x20, x17, x20, LSL #20\n"
    "madd x21, x21, x7, x13\n"
    "ld1w { z7.s }, p2/Z, [x14, #4, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "lsl x20, x20, #0x2\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "ld1w { z6.s }, p2/Z, [x14, #4, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "3:"  // Issue prefetches
    "subs x22, x22, #0x1\n"
    ".inst 0xf8b44abc  // rprfm pldstrm, x20, [x21]\n"
    "add x21, x21, %x[ld_in_col], LSL #2\n"
    "bgt 3b\n"
    "ldr x22, [%x[args], %[offsetof_Args_outptrs]]\n"
    "lsl x21, %x[ld_in_row], #0x2\n"
    ".inst 0xc0040f00  // mova za.d[x8, #0], { z24.d-z27.d }\n"
    "mov x10, #0x4\n"
    "ldr x20, [%x[args], %[offsetof_Args_ld_out_cols]]\n"
    "msub x13, x7, x21, x13\n"
    ".inst 0xc0040f01  // mova za.d[x8, #1], { z24.d-z27.d }\n"
    "ldr x21, [%x[args], %[offsetof_Args_pad_left]]\n"
    ".inst 0xc0040f02  // mova za.d[x8, #2], { z24.d-z27.d }\n"
    "ldp x9, x28, [x22], #0x10\n"
    ".inst 0xc0040f03  // mova za.d[x8, #3], { z24.d-z27.d }\n"
    "ldp x27, x26, [x20], #0x10\n"
    ".inst 0xc0040f04  // mova za.d[x8, #4], { z24.d-z27.d }\n"
    "ldp x25, x24, [x22], #0x10\n"
    "ldp x23, x22, [x20], #0x10\n"
    "cbz x21, 5f\n"
    "cmp x21, x10\n"
    "csel x20, x21, x10, LT\n"
    "sub x21, x21, x20\n"
    "sub x10, x10, x20\n"
    "cbz x21, 5f\n"
    ".inst 0xc0060c00  // mova { z0.d-z3.d }, za.d[x8, #0]\n"
    "sub x11, x11, x21\n"
    ".inst 0xc1bdcae0  // fclamp { z0.s-z3.s }, z23.s, z29.s\n"
    "4:"  // Left padding
    "subs x21, x21, #0x1\n"
    "st1w { z0.s }, p1, [x9]\n"
    "add x9, x9, x27, LSL #2\n"
    "st1w { z1.s }, p1, [x28]\n"
    "add x28, x28, x26, LSL #2\n"
    "st1w { z2.s }, p1, [x25]\n"
    "add x25, x25, x23, LSL #2\n"
    "st1w { z3.s }, p1, [x24]\n"
    "add x24, x24, x22, LSL #2\n"
    "bgt 4b\n"
    "5:"  // Left padding: End
    "adds XZR, x7, x6\n"
    "bne 12f\n"
    "cbz x10, 10f\n"
    "cmp x10, #0x1\n"
    "sub x15, x15, x10\n"
    "beq 9f\n"
    "cmp x10, #0x2\n"
    "beq 8f\n"
    "cmp x10, #0x3\n"
    "beq 7f\n"
    "6:"  // Unpadded: 4 priming loads
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "ld1w { z31.s }, p1/Z, [x13]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z0.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z1.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z2.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z3.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z4.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1351be0  // fmla za.s[x8, 0], { z31.s-z2.s }, z5.s\n"
    ".inst 0xa04049c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "ld1w { z5.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z6.s }, p1/Z, [x20]\n"
    ".inst 0xc13a1800  // fmla za.s[x8, 0], { z0.s-z3.s }, z10.s\n"
    ".inst 0xa04049cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1381820  // fmla za.s[x8, 0], { z1.s-z4.s }, z8.s\n"
    ".inst 0xa04049c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc13c1840  // fmla za.s[x8, 0], { z2.s-z5.s }, z12.s\n"
    ".inst 0xc1381860  // fmla za.s[x8, 0], { z3.s-z6.s }, z8.s\n"
    ".inst 0xa14049c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "7:"  // Unpadded: 3 priming loads
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "ld1w { z15.s }, p1/Z, [x13]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z16.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z17.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z18.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z19.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13d19e0  // fmla za.s[x8, 0], { z15.s-z18.s }, z13.s\n"
    "ld1w { z21.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13519e1  // fmla za.s[x8, 1], { z15.s-z18.s }, z5.s\n"
    ".inst 0xa14049c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "ld1w { z22.s }, p1/Z, [x20]\n"
    ".inst 0xc13b1a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z11.s\n"
    ".inst 0xc13a1a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z10.s\n"
    ".inst 0xa04049c2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc13d1a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z13.s\n"
    ".inst 0xc1351a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z5.s\n"
    ".inst 0xa14049c7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1331a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z3.s\n"
    ".inst 0xa04149c4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1321a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z2.s\n"
    ".inst 0xc13f1a60  // fmla za.s[x8, 0], { z19.s-z22.s }, z15.s\n"
    ".inst 0xc1371a61  // fmla za.s[x8, 1], { z19.s-z22.s }, z7.s\n"
    ".inst 0xa14049c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "8:"  // Unpadded: 2 priming loads
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "ld1w { z14.s }, p1/Z, [x13]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z15.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z16.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z17.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z18.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z19.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13419c0  // fmla za.s[x8, 0], { z14.s-z17.s }, z4.s\n"
    ".inst 0xa04149c2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13d19c1  // fmla za.s[x8, 1], { z14.s-z17.s }, z13.s\n"
    "ld1w { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13519c2  // fmla za.s[x8, 2], { z14.s-z17.s }, z5.s\n"
    ".inst 0xa14049c1  // ld1w { z1.s, z9.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "ld1w { z21.s }, p1/Z, [x20]\n"
    ".inst 0xc13819e0  // fmla za.s[x8, 0], { z15.s-z18.s }, z8.s\n"
    ".inst 0xa14149c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13b19e1  // fmla za.s[x8, 1], { z15.s-z18.s }, z11.s\n"
    ".inst 0xc13a19e2  // fmla za.s[x8, 2], { z15.s-z18.s }, z10.s\n"
    ".inst 0xa14049c6  // ld1w { z6.s, z14.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1321a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z2.s\n"
    ".inst 0xa04149ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1391a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z9.s\n"
    ".inst 0xc1311a02  // fmla za.s[x8, 2], { z16.s-z19.s }, z1.s\n"
    ".inst 0xa14049c7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1351a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z5.s\n"
    ".inst 0xc13e1a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z14.s\n"
    ".inst 0xc1361a22  // fmla za.s[x8, 2], { z17.s-z20.s }, z6.s\n"
    ".inst 0xc13a1a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z10.s\n"
    ".inst 0xa14149c4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13f1a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z15.s\n"
    ".inst 0xc1371a42  // fmla za.s[x8, 2], { z18.s-z21.s }, z7.s\n"
    ".inst 0xa14049c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "9:"  // Unpadded: 1 priming loads
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "ld1w { z15.s }, p1/Z, [x13]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z16.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z17.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z18.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z19.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13c19e0  // fmla za.s[x8, 0], { z15.s-z18.s }, z12.s\n"
    "ld1w { z21.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13419e1  // fmla za.s[x8, 1], { z15.s-z18.s }, z4.s\n"
    ".inst 0xa04149c2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13d19e2  // fmla za.s[x8, 2], { z15.s-z18.s }, z13.s\n"
    "ld1w { z22.s }, p1/Z, [x20]\n"
    ".inst 0xc13519e3  // fmla za.s[x8, 3], { z15.s-z18.s }, z5.s\n"
    ".inst 0xa14049c4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1391a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z9.s\n"
    ".inst 0xc1381a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z8.s\n"
    ".inst 0xa14149c7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13b1a02  // fmla za.s[x8, 2], { z16.s-z19.s }, z11.s\n"
    ".inst 0xc13a1a03  // fmla za.s[x8, 3], { z16.s-z19.s }, z10.s\n"
    ".inst 0xa04049c0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1331a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z3.s\n"
    ".inst 0xc1321a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z2.s\n"
    ".inst 0xa14149c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13c1a22  // fmla za.s[x8, 2], { z17.s-z20.s }, z12.s\n"
    ".inst 0xc1341a23  // fmla za.s[x8, 3], { z17.s-z20.s }, z4.s\n"
    ".inst 0xa14049c6  // ld1w { z6.s, z14.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc13f1a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z15.s\n"
    ".inst 0xc1371a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z7.s\n"
    ".inst 0xc1311a42  // fmla za.s[x8, 2], { z18.s-z21.s }, z1.s\n"
    "ld1w { z7.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc1301a43  // fmla za.s[x8, 3], { z18.s-z21.s }, z0.s\n"
    ".inst 0xc13d1a60  // fmla za.s[x8, 0], { z19.s-z22.s }, z13.s\n"
    ".inst 0xc1351a61  // fmla za.s[x8, 1], { z19.s-z22.s }, z5.s\n"
    ".inst 0xa14149c4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13e1a62  // fmla za.s[x8, 2], { z19.s-z22.s }, z14.s\n"
    ".inst 0xc1361a63  // fmla za.s[x8, 3], { z19.s-z22.s }, z6.s\n"
    ".inst 0xa14049c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "ld1w { z6.s }, p2/Z, [x14, #4, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "10:"  // Unpadded: 0 priming loads
    "cbz x15, 20f\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "ld1w { z14.s }, p1/Z, [x13]\n"
    "sub x15, x15, #0x1\n"
    "ld1w { z15.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "sub x11, x11, #0x1\n"
    "ld1w { z16.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "cmp x15, x11\n"
    "ld1w { z17.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "csel x21, x15, x11, LT\n"
    "ld1w { z18.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z19.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "sub x11, x11, x21\n"
    "ld1w { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z21.s }, p1/Z, [x20]\n"
    "cbz x21, 19f\n"
    "11:"  // Unpadded: Main loop
    ".inst 0xc13719c0  // fmla za.s[x8, 0], { z14.s-z17.s }, z7.s\n"
    "ld1w { z0.s }, p2/Z, [x14, #4, MUL VL]\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    "subs x21, x21, #0x1\n"
    ".inst 0xc13c19c1  // fmla za.s[x8, 1], { z14.s-z17.s }, z12.s\n"
    ".inst 0xc13419c2  // fmla za.s[x8, 2], { z14.s-z17.s }, z4.s\n"
    ".inst 0xa14149c4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13d19c3  // fmla za.s[x8, 3], { z14.s-z17.s }, z13.s\n"
    ".inst 0xc13619e0  // fmla za.s[x8, 0], { z15.s-z18.s }, z6.s\n"
    ".inst 0xc13519c4  // fmla za.s[x8, 4], { z14.s-z17.s }, z5.s\n"
    ".inst 0xa04049c2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc13919e1  // fmla za.s[x8, 1], { z15.s-z18.s }, z9.s\n"
    "ld1w { z6.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc13819e2  // fmla za.s[x8, 2], { z15.s-z18.s }, z8.s\n"
    ".inst 0xa14149c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1301a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z0.s\n"
    "ld1w { z14.s }, p1/Z, [x13]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    ".inst 0xc13b19e3  // fmla za.s[x8, 3], { z15.s-z18.s }, z11.s\n"
    ".inst 0xc13a19e4  // fmla za.s[x8, 4], { z15.s-z18.s }, z10.s\n"
    ".inst 0xa04049c0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc13c1a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z12.s\n"
    "ld1w { z8.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc1361a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z6.s\n"
    "ld1w { z15.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1341a02  // fmla za.s[x8, 2], { z16.s-z19.s }, z4.s\n"
    ".inst 0xa14149c4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1331a03  // fmla za.s[x8, 3], { z16.s-z19.s }, z3.s\n"
    ".inst 0xc1321a04  // fmla za.s[x8, 4], { z16.s-z19.s }, z2.s\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1381a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z8.s\n"
    "ld1w { z16.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13d1a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z13.s\n"
    ".inst 0xc1351a22  // fmla za.s[x8, 2], { z17.s-z20.s }, z5.s\n"
    "ld1w { z7.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc1311a23  // fmla za.s[x8, 3], { z17.s-z20.s }, z1.s\n"
    ".inst 0xc1301a24  // fmla za.s[x8, 4], { z17.s-z20.s }, z0.s\n"
    "ld1w { z17.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc0060c00  // mova { z0.d-z3.d }, za.d[x8, #0]\n"
    ".inst 0xc13c1a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z12.s\n"
    ".inst 0xc1341a42  // fmla za.s[x8, 2], { z18.s-z21.s }, z4.s\n"
    ".inst 0xa14149c4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13b1a43  // fmla za.s[x8, 3], { z18.s-z21.s }, z11.s\n"
    ".inst 0xc13a1a44  // fmla za.s[x8, 4], { z18.s-z21.s }, z10.s\n"
    "ld1w { z18.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "add x8, x8, #0x1\n"
    "ld1w { z19.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1bdcae0  // fclamp { z0.s-z3.s }, z23.s, z29.s\n"
    ".inst 0xc0040f04  // mova za.d[x8, #4], { z24.d-z27.d }\n"
    ".inst 0xa14049c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "ld1w { z20.s }, p1/Z, [x20]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "st1w { z0.s }, p1, [x9]\n"
    "add x9, x9, x27, LSL #2\n"
    "ld1w { z6.s }, p2/Z, [x14, #4, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "st1w { z1.s }, p1, [x28]\n"
    "add x28, x28, x26, LSL #2\n"
    "st1w { z2.s }, p1, [x25]\n"
    "add x25, x25, x23, LSL #2\n"
    "ld1w { z21.s }, p1/Z, [x20]\n"
    "st1w { z3.s }, p1, [x24]\n"
    "add x24, x24, x22, LSL #2\n"
    "bgt 11b\n"
    "b 19f\n"
    "12:"  // Padded
    "cbz x10, 17f\n"
    "cmp x10, #0x1\n"
    "sub x15, x15, x10\n"
    "beq 16f\n"
    "cmp x10, #0x2\n"
    "beq 15f\n"
    "cmp x10, #0x3\n"
    "beq 14f\n"
    "13:"  // Padded: 4 priming loads
    "mov x12, #0x0\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z14.s }, p0/Z, [x13]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z15.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    "ld1w { z17.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13519c0  // fmla za.s[x8, 0], { z14.s-z17.s }, z5.s\n"
    "ld1w { z18.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa04049c6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "ld1w { z19.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13a19e0  // fmla za.s[x8, 0], { z15.s-z18.s }, z10.s\n"
    ".inst 0xa14049c7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "ld1w { z20.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1361a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z6.s\n"
    ".inst 0xa14049c1  // ld1w { z1.s, z9.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    "ld1w { z21.s }, p0/Z, [x20]\n"
    ".inst 0xc1371a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z7.s\n"
    ".inst 0xc1311a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z1.s\n"
    ".inst 0xa14049c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "14:"  // Padded: 3 priming loads
    "mov x12, #0x0\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z15.s }, p0/Z, [x13]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z17.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    "ld1w { z18.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13d19e0  // fmla za.s[x8, 0], { z15.s-z18.s }, z13.s\n"
    "ld1w { z19.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13519e1  // fmla za.s[x8, 1], { z15.s-z18.s }, z5.s\n"
    ".inst 0xa14049c4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "ld1w { z20.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13b1a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z11.s\n"
    ".inst 0xc13a1a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z10.s\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "ld1w { z21.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13c1a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z12.s\n"
    ".inst 0xc1341a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z4.s\n"
    "ld1w { z22.s }, p0/Z, [x20]\n"
    ".inst 0xa04049cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc13f1a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z15.s\n"
    ".inst 0xc13e1a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z14.s\n"
    ".inst 0xa04149c4  // ld1w { z4.s-z5.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13d1a60  // fmla za.s[x8, 0], { z19.s-z22.s }, z13.s\n"
    ".inst 0xc13c1a61  // fmla za.s[x8, 1], { z19.s-z22.s }, z12.s\n"
    ".inst 0xa14049c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "15:"  // Padded: 2 priming loads
    "mov x12, #0x0\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z15.s }, p0/Z, [x13]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z17.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    "ld1w { z18.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13419e0  // fmla za.s[x8, 0], { z15.s-z18.s }, z4.s\n"
    "ld1w { z19.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13d19e1  // fmla za.s[x8, 1], { z15.s-z18.s }, z13.s\n"
    ".inst 0xa04149c0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13519e2  // fmla za.s[x8, 2], { z15.s-z18.s }, z5.s\n"
    ".inst 0xa04049c6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "ld1w { z20.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1381a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z8.s\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13b1a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z11.s\n"
    ".inst 0xc13a1a02  // fmla za.s[x8, 2], { z16.s-z19.s }, z10.s\n"
    "ld1w { z21.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa14049c2  // ld1w { z2.s, z10.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1301a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z0.s\n"
    ".inst 0xa04149cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1371a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z7.s\n"
    "ld1w { z22.s }, p0/Z, [x20]\n"
    ".inst 0xc1361a22  // fmla za.s[x8, 2], { z17.s-z20.s }, z6.s\n"
    ".inst 0xa04049c0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1381a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z8.s\n"
    ".inst 0xc13a1a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z10.s\n"
    ".inst 0xc1321a42  // fmla za.s[x8, 2], { z18.s-z21.s }, z2.s\n"
    ".inst 0xc13c1a60  // fmla za.s[x8, 0], { z19.s-z22.s }, z12.s\n"
    ".inst 0xa14149c4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1311a61  // fmla za.s[x8, 1], { z19.s-z22.s }, z1.s\n"
    ".inst 0xc1301a62  // fmla za.s[x8, 2], { z19.s-z22.s }, z0.s\n"
    ".inst 0xa14049c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "16:"  // Padded: 1 priming loads
    "mov x12, #0x0\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "ld1w { z14.s }, p0/Z, [x13]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "ld1w { z15.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    "ld1w { z17.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13c19c0  // fmla za.s[x8, 0], { z14.s-z17.s }, z12.s\n"
    "ld1w { z18.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13419c1  // fmla za.s[x8, 1], { z14.s-z17.s }, z4.s\n"
    ".inst 0xa14149c4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13d19c2  // fmla za.s[x8, 2], { z14.s-z17.s }, z13.s\n"
    ".inst 0xc13519c3  // fmla za.s[x8, 3], { z14.s-z17.s }, z5.s\n"
    "ld1w { z19.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13919e0  // fmla za.s[x8, 0], { z15.s-z18.s }, z9.s\n"
    ".inst 0xa04049c0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc13819e1  // fmla za.s[x8, 1], { z15.s-z18.s }, z8.s\n"
    ".inst 0xa04149c6  // ld1w { z6.s-z7.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13b19e2  // fmla za.s[x8, 2], { z15.s-z18.s }, z11.s\n"
    "ld1w { z20.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13a19e3  // fmla za.s[x8, 3], { z15.s-z18.s }, z10.s\n"
    ".inst 0xa04049ce  // ld1w { z14.s-z15.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc13c1a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z12.s\n"
    ".inst 0xc1341a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z4.s\n"
    "ld1w { z21.s }, p0/Z, [x20]\n"
    ".inst 0xc1311a02  // fmla za.s[x8, 2], { z16.s-z19.s }, z1.s\n"
    ".inst 0xa14149c4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1301a03  // fmla za.s[x8, 3], { z16.s-z19.s }, z0.s\n"
    ".inst 0xa04049c2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1371a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z7.s\n"
    ".inst 0xc1361a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z6.s\n"
    ".inst 0xc13f1a22  // fmla za.s[x8, 2], { z17.s-z20.s }, z15.s\n"
    "ld1w { z7.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc13e1a23  // fmla za.s[x8, 3], { z17.s-z20.s }, z14.s\n"
    ".inst 0xc13c1a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z12.s\n"
    ".inst 0xc1341a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z4.s\n"
    ".inst 0xa14149c4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1331a42  // fmla za.s[x8, 2], { z18.s-z21.s }, z3.s\n"
    ".inst 0xc1321a43  // fmla za.s[x8, 3], { z18.s-z21.s }, z2.s\n"
    ".inst 0xa14049c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "ld1w { z6.s }, p2/Z, [x14, #4, MUL VL]\n"
    "addvl x14, x14, #5\n"
    "17:"  // Padded: 0 priming loads
    "cbz x15, 20f\n"
    "mov x12, #0x0\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "sub x15, x15, #0x1\n"
    "sub x11, x11, #0x1\n"
    "cmp x15, x11\n"
    "ld1w { z14.s }, p0/Z, [x13]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "csel x21, x15, x11, LT\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    "sub x11, x11, x21\n"
    "ld1w { z15.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "mov x12, #0x4\n"
    "ld1w { z17.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z18.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z19.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z20.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "ld1w { z21.s }, p0/Z, [x20]\n"
    "cbz x21, 19f\n"
    "18:"  // Padded: Main loop
    ".inst 0xc13719c0  // fmla za.s[x8, 0], { z14.s-z17.s }, z7.s\n"
    "mov x12, #0x0\n"
    "ld1w { z7.s }, p2/Z, [x14, #4, MUL VL]\n"
    "add x20, x13, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13c19c1  // fmla za.s[x8, 1], { z14.s-z17.s }, z12.s\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "subs x21, x21, #0x1\n"
    ".inst 0xc13419c2  // fmla za.s[x8, 2], { z14.s-z17.s }, z4.s\n"
    ".inst 0xa14149c4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13d19c3  // fmla za.s[x8, 3], { z14.s-z17.s }, z13.s\n"
    ".inst 0xc13519c4  // fmla za.s[x8, 4], { z14.s-z17.s }, z5.s\n"
    "ld1w { z14.s }, p0/Z, [x13]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x13, x13, %x[ld_in_col], LSL #2\n"
    ".inst 0xc13619e0  // fmla za.s[x8, 0], { z15.s-z18.s }, z6.s\n"
    ".inst 0xa04049c0  // ld1w { z0.s-z1.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc13919e1  // fmla za.s[x8, 1], { z15.s-z18.s }, z9.s\n"
    "ld1w { z9.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc13819e2  // fmla za.s[x8, 2], { z15.s-z18.s }, z8.s\n"
    ".inst 0xa04149c2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13b19e3  // fmla za.s[x8, 3], { z15.s-z18.s }, z11.s\n"
    ".inst 0xc13a19e4  // fmla za.s[x8, 4], { z15.s-z18.s }, z10.s\n"
    "ld1w { z15.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1371a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z7.s\n"
    ".inst 0xa14049c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc13c1a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z12.s\n"
    "ld1w { z12.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc1341a02  // fmla za.s[x8, 2], { z16.s-z19.s }, z4.s\n"
    ".inst 0xa04149ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1311a03  // fmla za.s[x8, 3], { z16.s-z19.s }, z1.s\n"
    ".inst 0xc1301a04  // fmla za.s[x8, 4], { z16.s-z19.s }, z0.s\n"
    "ld1w { z16.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc1391a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z9.s\n"
    "mov x12, #0x4\n"
    ".inst 0xa14049c0  // ld1w { z0.s, z8.s }, pn10.b/Z, [x14]\n"
    "ldr x14, [%x[args], %[offsetof_Args_weights]]\n"
    ".inst 0xc1331a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z3.s\n"
    ".inst 0xc1321a22  // fmla za.s[x8, 2], { z17.s-z20.s }, z2.s\n"
    ".inst 0xc13d1a23  // fmla za.s[x8, 3], { z17.s-z20.s }, z13.s\n"
    ".inst 0xc1351a24  // fmla za.s[x8, 4], { z17.s-z20.s }, z5.s\n"
    "ld1w { z17.s }, p0/Z, [x20]\n"
    ".inst 0x25304500  // psel p0.s, p1.s/Z, p8.s[w12]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc13c1a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z12.s\n"
    "ld1w { z7.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc13b1a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z11.s\n"
    ".inst 0xc13a1a42  // fmla za.s[x8, 2], { z18.s-z21.s }, z10.s\n"
    ".inst 0xa14149c4  // ld1w { z4.s, z12.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1381a43  // fmla za.s[x8, 3], { z18.s-z21.s }, z8.s\n"
    ".inst 0xc1301a44  // fmla za.s[x8, 4], { z18.s-z21.s }, z0.s\n"
    "ld1w { z18.s }, p0/Z, [x20]\n"
    ".inst 0x25704500  // psel p0.s, p1.s/Z, p8.s[w12, #1]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xc0060c00  // mova { z0.d-z3.d }, za.d[x8, #0]\n"
    ".inst 0xa14049c5  // ld1w { z5.s, z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "add x8, x8, #0x1\n"
    ".inst 0xa04049ca  // ld1w { z10.s-z11.s }, pn10.b/Z, [x14]\n"
    ".inst 0xc0040f04  // mova za.d[x8, #4], { z24.d-z27.d }\n"
    "ld1w { z19.s }, p0/Z, [x20]\n"
    ".inst 0x25b04500  // psel p0.s, p1.s/Z, p8.s[w12, #2]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    "ld1w { z6.s }, p2/Z, [x14, #4, MUL VL]\n"
    "addvl x14, x14, #5\n"
    ".inst 0xc1bdcae0  // fclamp { z0.s-z3.s }, z23.s, z29.s\n"
    "ld1w { z20.s }, p0/Z, [x20]\n"
    ".inst 0x25f04500  // psel p0.s, p1.s/Z, p8.s[w12, #3]\n"
    "add x20, x20, %x[ld_in_row], LSL #2\n"
    "st1w { z0.s }, p1, [x9]\n"
    "add x9, x9, x27, LSL #2\n"
    "ld1w { z21.s }, p0/Z, [x20]\n"
    "st1w { z1.s }, p1, [x28]\n"
    "add x28, x28, x26, LSL #2\n"
    "st1w { z2.s }, p1, [x25]\n"
    "add x25, x25, x23, LSL #2\n"
    "st1w { z3.s }, p1, [x24]\n"
    "add x24, x24, x22, LSL #2\n"
    "bgt 18b\n"
    "19:"  // Main loop tail
    ".inst 0xc13719c0  // fmla za.s[x8, 0], { z14.s-z17.s }, z7.s\n"
    "ld1w { z7.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc13c19c1  // fmla za.s[x8, 1], { z14.s-z17.s }, z12.s\n"
    ".inst 0xc13419c2  // fmla za.s[x8, 2], { z14.s-z17.s }, z4.s\n"
    ".inst 0xa04149c2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13d19c3  // fmla za.s[x8, 3], { z14.s-z17.s }, z13.s\n"
    ".inst 0xc13619e0  // fmla za.s[x8, 0], { z15.s-z18.s }, z6.s\n"
    ".inst 0xc13519c4  // fmla za.s[x8, 4], { z14.s-z17.s }, z5.s\n"
    ".inst 0xa14049c6  // ld1w { z6.s, z14.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "ld1w { z0.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc13919e1  // fmla za.s[x8, 1], { z15.s-z18.s }, z9.s\n"
    ".inst 0xc13819e2  // fmla za.s[x8, 2], { z15.s-z18.s }, z8.s\n"
    ".inst 0xa04149c8  // ld1w { z8.s-z9.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc1371a00  // fmla za.s[x8, 0], { z16.s-z19.s }, z7.s\n"
    ".inst 0xc13b19e3  // fmla za.s[x8, 3], { z15.s-z18.s }, z11.s\n"
    ".inst 0xc13a19e4  // fmla za.s[x8, 4], { z15.s-z18.s }, z10.s\n"
    ".inst 0xa04049cc  // ld1w { z12.s-z13.s }, pn10.b/Z, [x14]\n"
    "addvl x14, x14, #5\n"
    "ld1w { z1.s }, p2/Z, [x14, #4, MUL VL]\n"
    ".inst 0xc1331a01  // fmla za.s[x8, 1], { z16.s-z19.s }, z3.s\n"
    ".inst 0xc1301a20  // fmla za.s[x8, 0], { z17.s-z20.s }, z0.s\n"
    ".inst 0xc1321a02  // fmla za.s[x8, 2], { z16.s-z19.s }, z2.s\n"
    ".inst 0xa14149c7  // ld1w { z7.s, z15.s }, pn10.b/Z, [x14, #0x2, MUL VL]\n"
    ".inst 0xc13e1a03  // fmla za.s[x8, 3], { z16.s-z19.s }, z14.s\n"
    ".inst 0xc1361a04  // fmla za.s[x8, 4], { z16.s-z19.s }, z6.s\n"
    ".inst 0xa04049c2  // ld1w { z2.s-z3.s }, pn10.b/Z, [x14]\n"
    ".inst 0xc1311a40  // fmla za.s[x8, 0], { z18.s-z21.s }, z1.s\n"
    ".inst 0xc1391a21  // fmla za.s[x8, 1], { z17.s-z20.s }, z9.s\n"
    ".inst 0xc1381a22  // fmla za.s[x8, 2], { z17.s-z20.s }, z8.s\n"
    ".inst 0xc13d1a23  // fmla za.s[x8, 3], { z17.s-z20.s }, z13.s\n"
    ".inst 0xc13c1a24  // fmla za.s[x8, 4], { z17.s-z20.s }, z12.s\n"
    ".inst 0xc0060c08  // mova { z8.d-z11.d }, za.d[x8, #0]\n"
    ".inst 0xc13f1a41  // fmla za.s[x8, 1], { z18.s-z21.s }, z15.s\n"
    ".inst 0xc1371a42  // fmla za.s[x8, 2], { z18.s-z21.s }, z7.s\n"
    ".inst 0xc1331a43  // fmla za.s[x8, 3], { z18.s-z21.s }, z3.s\n"
    ".inst 0xc1321a44  // fmla za.s[x8, 4], { z18.s-z21.s }, z2.s\n"
    "add x8, x8, #0x1\n"
    ".inst 0xc1bdcae8  // fclamp { z8.s-z11.s }, z23.s, z29.s\n"
    ".inst 0xc0040f04  // mova za.d[x8, #4], { z24.d-z27.d }\n"
    "st1w { z8.s }, p1, [x9]\n"
    "add x9, x9, x27, LSL #2\n"
    "st1w { z9.s }, p1, [x28]\n"
    "add x28, x28, x26, LSL #2\n"
    "st1w { z10.s }, p1, [x25]\n"
    "add x25, x25, x23, LSL #2\n"
    "st1w { z11.s }, p1, [x24]\n"
    "add x24, x24, x22, LSL #2\n"
    "20:"  // Main loop skip tail
    "cbz x11, 22f\n"
    "21:"  // Right padding loop
    ".inst 0xc0060c00  // mova { z0.d-z3.d }, za.d[x8, #0]\n"
    "add x8, x8, #0x1\n"
    "subs x11, x11, #0x1\n"
    ".inst 0xc0040f04  // mova za.d[x8, #4], { z24.d-z27.d }\n"
    ".inst 0xc1bdcae0  // fclamp { z0.s-z3.s }, z23.s, z29.s\n"
    "st1w { z0.s }, p1, [x9]\n"
    "add x9, x9, x27, LSL #2\n"
    "st1w { z1.s }, p1, [x28]\n"
    "add x28, x28, x26, LSL #2\n"
    "st1w { z2.s }, p1, [x25]\n"
    "add x25, x25, x23, LSL #2\n"
    "st1w { z3.s }, p1, [x24]\n"
    "add x24, x24, x22, LSL #2\n"
    "bgt 21b\n"
    "22:"  // End
    "ldr x20, [%x[args], %[offsetof_Args_weights]]\n"
    "incw x16\n"
    "whilelt p1.s, x16, x17\n"
    "incb x20, ALL, MUL #16\n"
    "incb x20, ALL, MUL #9\n"
    "str x20, [%x[args], %[offsetof_Args_weights]]\n"
    "ldr x21, [%x[args], %[offsetof_Args_ld_in_vl]]\n"
    "ldr x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "add x20, x20, x21, LSL #2\n"
    "str x20, [%x[args], %[offsetof_Args_inptr]]\n"
    "ldr x25, [%x[args], %[offsetof_Args_outptrs]]\n"
    "ldr x24, [%x[args], %[offsetof_Args_ld_out_vls]]\n"
    "ldp x23, x22, [x25, #0x0]\n"
    "ldp x21, x20, [x24, #0x0]\n"
    "add x23, x23, x21, LSL #2\n"
    "add x22, x22, x20, LSL #2\n"
    "stp x23, x22, [x25, #0x0]\n"
    "ldp x23, x22, [x25, #0x10]\n"
    "ldp x21, x20, [x24, #0x10]\n"
    "add x23, x23, x21, LSL #2\n"
    "add x22, x22, x20, LSL #2\n"
    "stp x23, x22, [x25, #0x10]\n"
    "b.any 1b\n"
    ".inst 0xd503467f  // SMSTOP\n"
    :
    : [args] "r" (&args), [ld_in_col] "r" (ld_in_col), [ld_in_row] "r" (ld_in_row), [offsetof_Args_bias] "I" (offsetof(Args, bias)), [offsetof_Args_clamp_max] "I" (offsetof(Args, clamp_max)), [offsetof_Args_clamp_min] "I" (offsetof(Args, clamp_min)), [offsetof_Args_current_channel] "I" (offsetof(Args, current_channel)), [offsetof_Args_inptr] "I" (offsetof(Args, inptr)), [offsetof_Args_input_cols] "I" (offsetof(Args, input_cols)), [offsetof_Args_ld_in_vl] "I" (offsetof(Args, ld_in_vl)), [offsetof_Args_ld_out_cols] "I" (offsetof(Args, ld_out_cols)), [offsetof_Args_ld_out_vls] "I" (offsetof(Args, ld_out_vls)), [offsetof_Args_n_channels] "I" (offsetof(Args, n_channels)), [offsetof_Args_outptrs] "I" (offsetof(Args, outptrs)), [offsetof_Args_output_cols] "I" (offsetof(Args, output_cols)), [offsetof_Args_pad_bottom] "I" (offsetof(Args, pad_bottom)), [offsetof_Args_pad_left] "I" (offsetof(Args, pad_left)), [offsetof_Args_pad_top] "I" (offsetof(Args, pad_top)), [offsetof_Args_weights] "I" (offsetof(Args, weights))
    : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "p5", "p6", "p7", "p8", "p9", "p10", "p11", "p12", "p13", "p14", "p15", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
  );
}

}  // namespace depthwise
}  // namespace arm_conv

#endif  // defined(ARM_COMPUTE_ENABLE_SME2)
