# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Unit tests for the ExternalDebsign task."""

import debusine.worker.tags as wtags
from debusine.artifacts.models import ArtifactCategory, DebianUpload
from debusine.server.tasks.wait.models import ExternalDebsignDynamicData
from debusine.server.workflows.models import WorkRequestWorkflowData
from debusine.tasks.server import ArtifactInfo
from debusine.tasks.tests.helper_mixin import FakeTaskDatabase
from debusine.test.django import TestCase


class ExternalDebsignTaskTests(TestCase):
    """Test the ExternalDebsign task."""

    def test_compute_system_required_tags(self) -> None:
        task = self.playground.create_wait_task(
            task_name="externaldebsign",
            task_data={"unsigned": 1},
            workflow_data=WorkRequestWorkflowData(needs_input=True),
        ).get_task()
        self.assertCountEqual(
            task.compute_system_required_tags(),
            [wtags.WORKER_TYPE_NOT_ASSIGNABLE],
        )

    def test_compute_dynamic_data(self) -> None:
        """Dynamic data receives relevant artifact IDs."""
        unsigned_lookup = "internal@collections/unsigned-upload"
        task_db = FakeTaskDatabase(
            single_lookups={
                # unsigned
                (unsigned_lookup, None): ArtifactInfo(
                    1,
                    ArtifactCategory.UPLOAD,
                    DebianUpload(
                        type="dpkg",
                        changes_fields={
                            "Architecture": "source",
                            "Files": [{"name": "hello_1.0.dsc"}],
                        },
                    ),
                ),
            }
        )

        task = self.playground.create_wait_task(
            task_name="externaldebsign",
            task_data={"unsigned": unsigned_lookup},
            workflow_data=WorkRequestWorkflowData(needs_input=True),
        ).get_task()
        self.assertEqual(
            task.compute_dynamic_data(task_db),
            ExternalDebsignDynamicData(unsigned_id=1),
        )

    def test_get_input_artifacts_ids(self) -> None:
        """Test get_input_artifacts_ids."""
        task = self.playground.create_wait_task(
            task_name="externaldebsign",
            task_data={"unsigned": 1},
            workflow_data=WorkRequestWorkflowData(needs_input=True),
        ).get_task()
        self.assertEqual(task.get_input_artifacts_ids(), [])

        task.dynamic_data = ExternalDebsignDynamicData(unsigned_id=1)
        self.assertEqual(task.get_input_artifacts_ids(), [1])

    def test_execute(self) -> None:
        """Executing the task does nothing, successfully."""
        work_request = self.playground.create_wait_task(
            task_name="externaldebsign",
            task_data={"unsigned": 1},
            workflow_data=WorkRequestWorkflowData(needs_input=True),
        )
        work_request.dynamic_task_data = {"unsigned_id": 1}
        work_request.save()
        task = work_request.get_task()
        self.assertTrue(task.execute())

    def test_label(self) -> None:
        """Test get_label."""
        work_request = self.playground.create_wait_task(
            task_name="externaldebsign",
            task_data={"unsigned": "internal@collections/unsigned-upload"},
            workflow_data=WorkRequestWorkflowData(needs_input=True),
        )
        work_request.dynamic_task_data = {"unsigned_id": 1}
        work_request.save()
        task = work_request.get_task()
        self.assertEqual(
            task.get_label(),
            "wait for external debsign for "
            "internal@collections/unsigned-upload",
        )
