/* This program is released under the Common Public License V1.0
 *
 * You should have received a copy of Common Public License V1.0 along with
 * with this program.
 */

/* Copyright IBM Corp. 2010, 2011 */
#include <fcntl.h>
#include <sys/errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ica_api.h"
#include "testcase.h"

#define BYTE 8

#define NUM_TESTS 12

#define AES_BLOCK_SIZE 16

unsigned int key_length[12] = {16, 16, 16, 16, 24, 24, 24, 24, 32, 32, 32,
				32};
unsigned char key[12][32] = {{
	0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15,
	0x88, 0x09, 0xcf, 0x4f, 0x3c},{
	0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15,
	0x88, 0x09, 0xcf, 0x4f, 0x3c},{
	0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15,
	0x88, 0x09, 0xcf, 0x4f, 0x3c},{
	0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15,
	0x88, 0x09, 0xcf, 0x4f, 0x3c},{
	0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10 ,0xf3,
	0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c,
	0x6b, 0x7b},{
	0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10 ,0xf3,
	0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c,
	0x6b, 0x7b},{
	0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10 ,0xf3,
	0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c,
	0x6b, 0x7b},{
	0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10 ,0xf3,
	0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c,
	0x6b, 0x7b},{
	0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae,
	0xf0, 0x85, 0x7d, 0x77, 0x81, 0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61,
	0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4},{
	0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae,
	0xf0, 0x85, 0x7d, 0x77, 0x81, 0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61,
	0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4},{
	0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae,
	0xf0, 0x85, 0x7d, 0x77, 0x81, 0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61,
	0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4},{
	0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae,
	0xf0, 0x85, 0x7d, 0x77, 0x81, 0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61,
	0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4}
};

unsigned char last_block[3][16] = {{
	0x7d, 0xf7, 0x6b, 0x0c, 0x1a, 0xb8, 0x99, 0xb3, 0x3e, 0x42, 0xf0,
	0x47, 0xb9, 0x1b, 0x54, 0x6f},{
	0x22, 0x45, 0x2d, 0x8e, 0x49, 0xa8, 0xa5, 0x93, 0x9f, 0x73, 0x21,
	0xce, 0xea, 0x6d, 0x51, 0x4b},{
	0xe5, 0x68, 0xf6, 0x81, 0x94, 0xcf, 0x76, 0xd6, 0x17, 0x4d, 0x4c,
	0xc0, 0x43, 0x10, 0xa8, 0x54}
};

unsigned long mlen[12] = { 0, 16, 40, 64, 0,16, 40, 64, 0, 16, 40, 64};
unsigned char message[12][512] = {{
	0x00},{
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e,
	0x11, 0x73, 0x93, 0x17, 0x2a},{
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e,
	0x11, 0x73, 0x93, 0x17, 0x2a, 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03,
	0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf ,0x8e, 0x51, 0x30,
	0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11},{
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e,
	0x11, 0x73, 0x93, 0x17, 0x2a, 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03,
	0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51, 0x30,
	0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19,
	0x1a, 0x0a, 0x52, 0xef, 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b,
	0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},{
	0x00},{
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e,
	0x11, 0x73, 0x93, 0x17, 0x2a},{
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e,
	0x11, 0x73, 0x93, 0x17, 0x2a, 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03,
	0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf ,0x8e, 0x51, 0x30,
	0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11},{
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e,
	0x11, 0x73, 0x93, 0x17, 0x2a, 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03,
	0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51, 0x30,
	0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19,
	0x1a, 0x0a, 0x52, 0xef, 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b,
	0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},{
	0x00},{
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e,
	0x11, 0x73, 0x93, 0x17, 0x2a},{
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e,
	0x11, 0x73, 0x93, 0x17, 0x2a, 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03,
	0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf ,0x8e, 0x51, 0x30,
	0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11},{
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e,
	0x11, 0x73, 0x93, 0x17, 0x2a, 0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03,
	0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51, 0x30,
	0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19,
	0x1a, 0x0a, 0x52, 0xef, 0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b,
	0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10}
};

unsigned char expected_cmac[12][16] = {{
	0xbb, 0x1d, 0x69, 0x29, 0xe9, 0x59, 0x37, 0x28, 0x7f, 0xa3, 0x7d,
	0x12, 0x9b, 0x75, 0x67, 0x46},{
	0x07, 0x0a, 0x16, 0xb4, 0x6b, 0x4d, 0x41, 0x44, 0xf7, 0x9b, 0xdd,
	0x9d, 0xd0, 0x4a, 0x28, 0x7c},{
	0xdf, 0xa6, 0x67, 0x47, 0xde, 0x9a, 0xe6, 0x30, 0x30, 0xca, 0x32,
	0x61, 0x14, 0x97, 0xc8, 0x27},{
	0x51, 0xf0, 0xbe, 0xbf, 0x7e, 0x3b, 0x9d, 0x92, 0xfc, 0x49, 0x74,
	0x17, 0x79, 0x36, 0x3c, 0xfe},{
	0xd1, 0x7d, 0xdf, 0x46, 0xad, 0xaa, 0xcd, 0xe5, 0x31, 0xca, 0xc4,
	0x83, 0xde, 0x7a, 0x93, 0x67},{
	0x9e, 0x99, 0xa7, 0xbf, 0x31, 0xe7, 0x10, 0x90, 0x06, 0x62, 0xf6,
	0x5e, 0x61, 0x7c, 0x51, 0x84},{
	0x8a, 0x1d, 0xe5, 0xbe, 0x2e, 0xb3, 0x1a, 0xad, 0x08, 0x9a, 0x82,
	0xe6, 0xee, 0x90, 0x8b, 0x0e},{
	0xa1, 0xd5, 0xdf, 0x0e, 0xed, 0x79, 0x0f, 0x79, 0x4d, 0x77, 0x58,
	0x96, 0x59, 0xf3, 0x9a, 0x11},{
	0x02, 0x89, 0x62, 0xf6, 0x1b, 0x7b, 0xf8, 0x9e, 0xfc, 0x6b, 0x55,
	0x1f, 0x46, 0x67, 0xd9, 0x83},{
	0x28, 0xa7, 0x02, 0x3f, 0x45, 0x2e, 0x8f, 0x82, 0xbd, 0x4b, 0xf2,
	0x8d, 0x8c, 0x37, 0xc3, 0x5c},{
	0xaa, 0xf3, 0xd8, 0xf1, 0xde, 0x56, 0x40, 0xc2, 0x32, 0xf5, 0xb1,
	0x69, 0xb9, 0xc9, 0x11, 0xe6},{
	0xe1, 0x99, 0x21, 0x90, 0x54, 0x9f, 0x6e, 0xd5, 0x69, 0x6a, 0x2c,
	0x05, 0x6c, 0x31, 0x54, 0x10}
};

unsigned int ica_aes_cmac_chaining(unsigned char *, unsigned long,
				   unsigned char *, unsigned int,
				   unsigned char *, unsigned int,
				   unsigned int);

unsigned int i = 0;

unsigned char *cmac;
unsigned int cmac_length = 16;

int api_cmac_test(void)
{
	int rc = 0;

	VV_(printf("Test of CMAC api\n"));
	for (i = 0 ; i < NUM_TESTS; i++) {
		if (!(cmac = malloc(cmac_length)))
			return EINVAL;
		memset(cmac, 0, cmac_length);
		rc = (ica_aes_cmac(message[i], mlen[i],
				   cmac, cmac_length,
				   key[i], key_length[i],
				   ICA_ENCRYPT));
		if (rc) {
			VV_(printf("ica_aes_cmac generate failed with errno %d (0x%x)."
				"\n",rc,rc));
			return rc;
		}
		if (memcmp(cmac, expected_cmac[i], cmac_length) != 0) {
			VV_(printf("This does NOT match the known result. "
				"Testcase %i failed\n",i));
			VV_(printf("\nOutput MAC for test %d:\n", i));
			dump_array((unsigned char *)cmac, cmac_length);
			VV_(printf("\nExpected MAC for test %d:\n", i));
			dump_array((unsigned char *)expected_cmac[i], 16);
			free(cmac);
			return 1;
		}
		VV_(printf("Expected MAC has been generated.\n"));
		rc = (ica_aes_cmac(message[i], mlen[i],
				   cmac, cmac_length,
				   key[i], key_length[i],
				   ICA_DECRYPT));
		if (rc) {
			VV_(printf("ica_aes_cmac verify failed with errno %d (0x%x).\n",
				rc, rc));
			free(cmac);
			return rc;
		}
		free(cmac);
		if (!rc) {
			VV_(printf("MAC was successful verified. Test %i "
			"succeeded\n",i));
		}
		else {
			printf("MAC verification failed for test %i "
				"with RC=%i\n",i,rc);
			return rc;
		}
	}
	return 0;
}


inline unsigned int ica_aes_cmac_chaining(unsigned char *in,
					  unsigned long in_length,
					  unsigned char *cmac,
					  unsigned int cmac_length,
					  unsigned char *key,
					  unsigned int key_length,
					  unsigned int direction)
{
	unsigned char *last_block = NULL;
	unsigned long tmp_length;
	unsigned char tmp_iv[AES_BLOCK_SIZE];
	unsigned int rc;

	memset(tmp_iv, 0x00, AES_BLOCK_SIZE);

	for (last_block = in, tmp_length = in_length;
	     tmp_length > (AES_BLOCK_SIZE);
	     last_block += AES_BLOCK_SIZE, tmp_length -= AES_BLOCK_SIZE) {
		rc = ica_aes_cmac_intermediate(last_block, AES_BLOCK_SIZE,
					       key, key_length,
					       tmp_iv);
		if (rc)
			return rc;
	}

	return ica_aes_cmac_last(last_block, tmp_length,
				 cmac, cmac_length,
				 key, key_length,
				 tmp_iv,
				 direction);
}

int api_cmac_chaining_test(void)
{
	int rc = 0;

	VV_(printf("Test of CMAC chaining api\n"));
	for (i = 0 ; i < NUM_TESTS; i++) {
		if (!(cmac = malloc(cmac_length)))
			return EINVAL;
		memset(cmac, 0, cmac_length);
		rc = ica_aes_cmac_chaining(message[i], mlen[i],
					   cmac, cmac_length,
					   key[i], key_length[i],
					   ICA_ENCRYPT);
		if (rc) {
			VV_(printf("ica_aes_cmac chaining generate failed with errno %d (0x%x)."
				"\n",rc,rc));
			return rc;
		}
		if (memcmp(cmac, expected_cmac[i], cmac_length) != 0) {
			VV_(printf("This does NOT match the known result. "
				"Testcase %i failed\n",i));
			VV_(printf("\nOutput MAC for test %d:\n", i));
			dump_array((unsigned char *)cmac, cmac_length);
			VV_(printf("\nExpected MAC for test %d:\n", i));
			dump_array((unsigned char *)expected_cmac[i], 16);
			free(cmac);
			return 1;
		}
		VV_(printf("Expected MAC has been generated.\n"));
		rc = ica_aes_cmac_chaining(message[i], mlen[i],
					   cmac, cmac_length,
					   key[i], key_length[i],
					   ICA_DECRYPT);
		if (rc) {
			VV_(printf("ica_aes_cmac verify failed with errno %d (0x%x).\n",
				rc, rc));
			free(cmac);
			return rc;
		}
		free(cmac);
		if (!rc) {
			VV_(printf("MAC was successful verified. Test %i "
				"succeeded\n",i));
		}
		else {
			VV_(printf("MAC verification failed for test %i "
				"with RC=%i\n",i,rc));
			return rc;
		}
	}
	return 0;
}

int main(int argc, char **argv)
{
	int rc = 0;

	set_verbosity(argc, argv);

	rc = api_cmac_test();
	if (rc) {
		printf("api_cmac_test failed with rc = %i\n", rc);
		return rc;
	}

	rc = api_cmac_chaining_test();
	if (rc) {
		printf("api_cmac_test failed with rc = %i\n", rc);
		return rc;
	}

	printf("All CMAC tests passed.\n");
	return 0;
}
