/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright (C) 2009--2020 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * *****************************************************************************
 * This specific code is a port to C++ of the Envemind Python code by Radzinski
 * and colleagues of IsoSpec fame (Lacki, Startek and company :-)
 *
 * See https://github.com/PiotrRadzinski/envemind.
 * *****************************************************************************
 *
 * END software license
 */


#include <QDebug>
#include <QDateTime>
#include <QtNetwork>

#include "MsXpS/libXpertMassCore/MassDataServer.hpp"

namespace MsXpS
{
namespace libXpertMassCore
{


/*!
\class MsXpS::libXpertMassCore::MassDataServer
\inmodule libXpertMassCore
\ingroup XpertMassCoreUtilities
\inheaderfile MassDataServer.hpp

\brief The MassDataServer class provides a network server.
*/

/*!
\variable MsXpS::libXpertMassCore::MassDataServer::m_ipAddress

\brief The IP address at which this server serves data.
*/

/*!
 * \variable MsXpS::libXpertMassCore::MassDataServer::m_portNumber
 *
 * \brief The port number at which this server serves data.
 */

/*!
 * \variable MsXpS::libXpertMassCore::MassDataServer::m_clients
 *
 * \brief The list of QTcpSocket instances representing the clients connected to
 * this server.
 */

/*!
 * \variable MsXpS::libXpertMassCore::MassDataServer::m_readyClients
 *
 * \brief The list of QTcpSocket instances representing the clients connected to
 * this server and effectively expecting data from this server.
 */

/*!
\brief Constructs a MassDataServer instance.

\list
\li \a parent: QObject parent.
\endlist
*/
MassDataServer::MassDataServer(QObject *parent): QTcpServer(parent)
{
}

/*!
\brief Destructs this MassDataServer instance.
*/
MassDataServer::~MassDataServer()
{
}

bool
MassDataServer::start()
{
  if(!listen())
    {
      qDebug() << "Failed to start the server.";
      return false;
    }

  // At this point try to get to the details.

  QList<QHostAddress> ip_addressesList = QNetworkInterface::allAddresses();

  // Use the first non-localhost IPv4 address
  for(int i = 0; i < ip_addressesList.size(); ++i)
    {
      if(ip_addressesList.at(i) != QHostAddress::LocalHost &&
         ip_addressesList.at(i).toIPv4Address())
        {
          m_ipAddress = ip_addressesList.at(i).toString();
          break;
        }
    }

  // If we did not find one, use IPv4 localhost
  if(m_ipAddress.isEmpty())
    m_ipAddress = QHostAddress(QHostAddress::LocalHost).toString();

  m_portNumber = serverPort();

  return true;
}

QString
MassDataServer::getIpAddress() const
{
  return m_ipAddress;
}

int
MassDataServer::getPortNumber() const
{
  return m_portNumber;
}

/*!
\brief Sets to this MassDataServer instance the data to be served in
\a byte_array_to_serve.
*/
void
MassDataServer::serveData(const QByteArray &byte_array_to_serve)
{
  if(!byte_array_to_serve.size())
    return;

  for(QTcpSocket *client_p : m_readyClients)
    {
      if(client_p->state() == QAbstractSocket::ConnectedState)
        {
          QByteArray byte_array;
          QDataStream out_stream(&byte_array, QIODevice::WriteOnly);
          out_stream.setVersion(QDataStream::Qt_5_0);
          out_stream << byte_array_to_serve;

          qDebug() << "On the verge of writing byte_array of size:"
                   << byte_array.size();

          int written_bytes = client_p->write(byte_array);
          client_p->flush();

          qDebug() << "Now written " << written_bytes
                   << " bytes to the socket at" << QDateTime::currentDateTime();

          if(written_bytes >= byte_array.size())
            {
              qDebug() << "Data successfully written.";

              emit writtenDataSignal(written_bytes);
            }
          else
            {
              qWarning() << "The data written to the socket were less than the "
                            "initial data.";
            }
        }
    }
}

/*!
\brief Handles an incoming connection with socket descriptor \a
socket_descriptor.

If the connection is effective, that is, if a QTcpSocket client
could be allocated and configured with \a socket_descriptor, that client is
appended to the \l{m_clients} list of clients.
*/
void
MassDataServer::incomingConnection(qintptr socket_descriptor)
{

  // It is not because we have not data to use as a response to the caller
  // that we do not perform the connection and then closing stuff! Otherwise we
  // consume a file descriptor (the socket) each time a connection is tried
  // here from the client.... Bug that has broken my head for weeks...

  QTcpSocket *client_tcp_socket_p = new QTcpSocket(this);

  if(!client_tcp_socket_p->setSocketDescriptor(socket_descriptor))
    {
      delete client_tcp_socket_p;
      return;
    }

  qDebug() << "MineXpert3 connected from"
           << client_tcp_socket_p->peerAddress().toString() << ":"
           << client_tcp_socket_p->peerPort();

  connect(client_tcp_socket_p,
          &QTcpSocket::disconnected,
          this,
          [this, client_tcp_socket_p]() {
            qDebug() << "One client has disconnected, removing it.";

            qDebug() << "Before removing, there are" << m_readyClients.size()
                     << "ready clients.";
            m_readyClients.remove(client_tcp_socket_p);
            qDebug() << "Now remaining" << m_readyClients.size()
                     << "ready clients.";

            qDebug() << "Before removing, there are" << m_clients.size()
                     << "clients.";
            m_clients.removeOne(client_tcp_socket_p);
            qDebug() << "Now remaining" << m_clients.size() << "clients.";

            client_tcp_socket_p->deleteLater();
          });

  // When the client connects, it sends a "READY" message that we catch
  // to unambiguously determine that the client is connected.
  connect(client_tcp_socket_p,
          &QTcpSocket::readyRead,
          this,
          [this, client_tcp_socket_p]() {
            QByteArray msg = client_tcp_socket_p->readAll();

            if(msg.contains("READY"))
              {
                qDebug() << "MineXpert3 is ready!";
                m_readyClients.insert(client_tcp_socket_p);
              }
          });

  m_clients.append(client_tcp_socket_p);

  qDebug() << "MineXpert3 connected";

  m_clients.append(client_tcp_socket_p);
}

/*!
\brief Returns true if this server has clients (\l{MassDataClient}) that are
ready to receive data.
*/
bool
MassDataServer::hasReadyClient() const
{
  return !m_readyClients.isEmpty();
}

/*!
\brief Reports the \a socket_error to the console using qDebug().
*/
void
MassDataServer::error(QTcpSocket::SocketError socket_error)
{
  qDebug() << "An error occurred in the mass data server:" << socket_error;
}


} // namespace libXpertMassCore

} // namespace MsXpS
