// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using Test.Cryptography;

namespace System.Security.Cryptography.X509Certificates.Tests
{
    internal static class TestData
    {
        public static byte[] MsCertificate = (
            "308204ec308203d4a003020102021333000000b011af0a8bd03b9fdd00010000" +
            "00b0300d06092a864886f70d01010505003079310b3009060355040613025553" +
            "311330110603550408130a57617368696e67746f6e3110300e06035504071307" +
            "5265646d6f6e64311e301c060355040a13154d6963726f736f667420436f7270" +
            "6f726174696f6e312330210603550403131a4d6963726f736f667420436f6465" +
            "205369676e696e6720504341301e170d3133303132343232333333395a170d31" +
            "34303432343232333333395a308183310b300906035504061302555331133011" +
            "0603550408130a57617368696e67746f6e3110300e060355040713075265646d" +
            "6f6e64311e301c060355040a13154d6963726f736f667420436f72706f726174" +
            "696f6e310d300b060355040b13044d4f5052311e301c060355040313154d6963" +
            "726f736f667420436f72706f726174696f6e30820122300d06092a864886f70d" +
            "01010105000382010f003082010a0282010100e8af5ca2200df8287cbc057b7f" +
            "adeeeb76ac28533f3adb407db38e33e6573fa551153454a5cfb48ba93fa837e1" +
            "2d50ed35164eef4d7adb137688b02cf0595ca9ebe1d72975e41b85279bf3f82d" +
            "9e41362b0b40fbbe3bbab95c759316524bca33c537b0f3eb7ea8f541155c0865" +
            "1d2137f02cba220b10b1109d772285847c4fb91b90b0f5a3fe8bf40c9a4ea0f5" +
            "c90a21e2aae3013647fd2f826a8103f5a935dc94579dfb4bd40e82db388f12fe" +
            "e3d67a748864e162c4252e2aae9d181f0e1eb6c2af24b40e50bcde1c935c49a6" +
            "79b5b6dbcef9707b280184b82a29cfbfa90505e1e00f714dfdad5c238329ebc7" +
            "c54ac8e82784d37ec6430b950005b14f6571c50203010001a38201603082015c" +
            "30130603551d25040c300a06082b06010505070303301d0603551d0e04160414" +
            "5971a65a334dda980780ff841ebe87f9723241f230510603551d11044a3048a4" +
            "463044310d300b060355040b13044d4f5052313330310603550405132a333135" +
            "39352b34666166306237312d616433372d346161332d613637312d3736626330" +
            "35323334346164301f0603551d23041830168014cb11e8cad2b4165801c9372e" +
            "331616b94c9a0a1f30560603551d1f044f304d304ba049a0478645687474703a" +
            "2f2f63726c2e6d6963726f736f66742e636f6d2f706b692f63726c2f70726f64" +
            "756374732f4d6963436f645369675043415f30382d33312d323031302e63726c" +
            "305a06082b06010505070101044e304c304a06082b06010505073002863e6874" +
            "74703a2f2f7777772e6d6963726f736f66742e636f6d2f706b692f6365727473" +
            "2f4d6963436f645369675043415f30382d33312d323031302e637274300d0609" +
            "2a864886f70d0101050500038201010031d76e2a12573381d59dc6ebf93ad444" +
            "4d089eee5edf6a5bb779cf029cbc76689e90a19c0bc37fa28cf14dba9539fb0d" +
            "e0e19bf45d240f1b8d88153a7cdbadceb3c96cba392c457d24115426300d0dff" +
            "47ea0307e5e4665d2c7b9d1da910fa1cb074f24f696b9ea92484daed96a0df73" +
            "a4ef6a1aac4b629ef17cc0147f48cd4db244f9f03c936d42d8e87ce617a09b68" +
            "680928f90297ef1103ba6752adc1e9b373a6d263cd4ae23ee4f34efdffa1e0bb" +
            "02133b5d20de553fa3ae9040313875285e04a9466de6f57a7940bd1fcde845d5" +
            "aee25d3ef575c7e6666360ccd59a84878d2430f7ef34d0631db142674a0e4bbf" +
            "3a0eefb6953aa738e4259208a6886682").HexToByteArray();

        public static readonly byte[] MsCertificatePemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN CERTIFICATE-----
MIIE7DCCA9SgAwIBAgITMwAAALARrwqL0Duf3QABAAAAsDANBgkqhkiG9w0BAQUF
ADB5MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSMwIQYDVQQD
ExpNaWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzlaFw0x
NDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNyb3NvZnQgQ29ycG9yYXRp
b24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/
re7rdqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01Fk7vTXrbE3aIsCzw
WVyp6+HXKXXkG4Unm/P4LZ5BNisLQPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhl
HSE38Cy6IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMBNkf9L4JqgQP1
qTXclFed+0vUDoLbOI8S/uPWenSIZOFixCUuKq6dGB8OHrbCryS0DlC83hyTXEmm
ebW22875cHsoAYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZDC5UABbFP
ZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAKBggrBgEFBQcDAzAdBgNVHQ4EFgQU
WXGmWjNN2pgHgP+EHr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsTBE1P
UFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFkMzctNGFhMy1hNjcxLTc2YmMw
NTIzNDRhZDAfBgNVHSMEGDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9k
dWN0cy9NaWNDb2RTaWdQQ0FfMDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBM
MEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRz
L01pY0NvZFNpZ1BDQV8wOC0zMS0yMDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEA
MdduKhJXM4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/oozxTbqVOfsN
4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9JBFUJjANDf9H6gMH5eRmXSx7nR2pEPoc
sHTyT2lrnqkkhNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjofOYXoJto
aAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO/f+h4LsCEztdIN5VP6OukEAxOHUo
XgSpRm3m9Xp5QL0fzehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdKDku/
Og7vtpU6pzjkJZIIpohmgg==
-----END CERTIFICATE-----
");

        // 'cert.pem' generated using 'openssl req -x509 -newkey rsa:2048 -keyout key.pem -out cert.pem -days 365'
        public static readonly byte[] SelfSigned1PemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN CERTIFICATE-----
MIIDWjCCAkKgAwIBAgIJAJpCQ7mtFWHeMA0GCSqGSIb3DQEBCwUAMEIxCzAJBgNV
BAYTAlhYMRUwEwYDVQQHDAxEZWZhdWx0IENpdHkxHDAaBgNVBAoME0RlZmF1bHQg
Q29tcGFueSBMdGQwHhcNMTgwNTMwMTgyNjM1WhcNMTkwNTMwMTgyNjM1WjBCMQsw
CQYDVQQGEwJYWDEVMBMGA1UEBwwMRGVmYXVsdCBDaXR5MRwwGgYDVQQKDBNEZWZh
dWx0IENvbXBhbnkgTHRkMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
pfYZTHjzei9U3QxiIIjESsf9z3Bfl8FAQLIU+OeICN3upnDvTgeWM/Jw7LwiuHhu
XvSawPwQ8ONvUeSG/wfyjYyTB7VBpVnNi6oTR6E1WSuiu0iT3qlDHVwArTI5DvIM
FzP3/AT1Ub5SvwVbWiR2za6wuUIsryyLz5+zCwGr+J/Xbmta/H9IT9NLwmDJCZQe
4Q4hCWhf7FKdXWt59y9PofVnE7R8CKNfUKr6GA+gy+SEtM/cHgqox5PErnV9b14U
uVROnRUyo1bFwTOdoW3zf5S4VZ4pFPJHNYACHEMiE0eNgfJf+QeyPUPN50neEAbf
kQYkeEET8dW6JlDFrAI4wwIDAQABo1MwUTAdBgNVHQ4EFgQUK+C/eGYPlV+KaTvj
tF6lJaKmo3EwHwYDVR0jBBgwFoAUK+C/eGYPlV+KaTvjtF6lJaKmo3EwDwYDVR0T
AQH/BAUwAwEB/zANBgkqhkiG9w0BAQsFAAOCAQEAZUjvDMhGc45TLRHKO5rsyifN
g7qb3dO5vtD/JWeo+wyMYcBHIANIIxYrkT0dRBQWQerVDBvsAESahM3f0SdszGac
6y1qxQWxfjxRiCwrEQ7JVZkmspYLbOxaS1T2IZUo3D7VJReyna6r11EKy7i49Toa
KmrhTLBsHV+MUgPRtupiOOu0fXqfxpXE7XEvi0hyv8PKli+Oww2Zyt1jTTvv2RTA
eJRqTUNUbWEDesXAOh5CY6Xjfg7Gt6IYQHt0JMw29pXB3TV2uyXuvFNsc725cPbW
JCuC9TGQRUAUj+LZ43tTrfaZ7g5L80/eRrvlx5MIJSsX8cev8pZYx224WRtk/w==
-----END CERTIFICATE-----
");

        // 'cert.pem' generated using 'openssl req -x509 -newkey rsa:2048 -keyout key.pem -out cert.pem -days 365'
        public static readonly byte[] SelfSigned2PemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN CERTIFICATE-----
MIIDWjCCAkKgAwIBAgIJAM6YQ4PrC9jaMA0GCSqGSIb3DQEBCwUAMEIxCzAJBgNV
BAYTAlhYMRUwEwYDVQQHDAxEZWZhdWx0IENpdHkxHDAaBgNVBAoME0RlZmF1bHQg
Q29tcGFueSBMdGQwHhcNMTgwNTMwMTgyNjQ4WhcNMTkwNTMwMTgyNjQ4WjBCMQsw
CQYDVQQGEwJYWDEVMBMGA1UEBwwMRGVmYXVsdCBDaXR5MRwwGgYDVQQKDBNEZWZh
dWx0IENvbXBhbnkgTHRkMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
7vkM6zrhXJFtrV63lUb4fsjZG2JYvSRGYv4Y/rwe7VLVdTYvMjosyvKCHJ4Frmtb
YU4jJeB+859mQAd3IOBEhgUJuJ6gC8cOJAwUFJNUabeuafXG2zw/U+396csRKr11
iBUpvooFJR7KLWrqPKXhK5yESV1k7OzSSZs4owmyIvSaGQO2T63S39OYJhq8ZUlO
+MznaOQGp2J+JWncZo9XCpiotZwdNtw5k+F1g3NAz4/+Vkah/SfQhcNCfJyfVDCX
IwBS+gz9i1BIw6s+SLYtkp167yyizmVIWoXtkgCPaeG0FzBPAhL9GDLTItJ/V/O5
F9SjbvS+4rUIuPSn7NdodwIDAQABo1MwUTAdBgNVHQ4EFgQUq4v4TrvYrsbKDRGF
bMnj3++P9B4wHwYDVR0jBBgwFoAUq4v4TrvYrsbKDRGFbMnj3++P9B4wDwYDVR0T
AQH/BAUwAwEB/zANBgkqhkiG9w0BAQsFAAOCAQEAS4ZKEGVfIrHMmYAZe9p3jm15
85OIPLlM4q6QjRccLJ4t2+xYLRU9PNa2Qmz8l+SFOy9XN9yJp79lSi1gN4lJ8ZdI
kwu+zPEzwsJnb6f3zRu2RQUeAkaShDCEdxpyKHEky1KVG2nOa3cKp+pqzN4DQ3Rp
cJCjcP5ncNJ0bbCZTS7w0oVvX5JhBWIigw3CN5rL2rf83CTPPBzUype0bt97sBSs
dxIPtH9l/q9OgdaCrPE8KBqcwXsfNlFwYGjkqmN/v7WaysBRdblHcoWmry3YsaK2
/tZo6lmYOHpdqL0OdDwlldToY7QdL1coICfHas5Ony49OHTCUZz6G/AS+3a3gQ==
-----END CERTIFICATE-----
");

        // 'cert.pem' generated using 'openssl req -x509 -newkey rsa:2048 -keyout key.pem -out cert.pem -days 365'
        public static readonly byte[] SelfSigned3PemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN CERTIFICATE-----
MIIDWjCCAkKgAwIBAgIJANzv9IQvr0bwMA0GCSqGSIb3DQEBCwUAMEIxCzAJBgNV
BAYTAlhYMRUwEwYDVQQHDAxEZWZhdWx0IENpdHkxHDAaBgNVBAoME0RlZmF1bHQg
Q29tcGFueSBMdGQwHhcNMTgwNjA0MTMzMjIxWhcNMTkwNjA0MTMzMjIxWjBCMQsw
CQYDVQQGEwJYWDEVMBMGA1UEBwwMRGVmYXVsdCBDaXR5MRwwGgYDVQQKDBNEZWZh
dWx0IENvbXBhbnkgTHRkMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
wy+py+hFxSmCGTZmHrQm1Yobzxf34l+J8VD33ObGV1qIFFulxz8pnUU4gKf6FQNU
wAbezJ78Eqsjt4c7mwnGTdavWSZyDJ136bQzn52wsTOGRfUBe1vt+SMy7h8Nhhf3
ejRHQVsZKNfiGOekmjBKFLliavo6I8j80UsmpvAJ+TWnYpVQBf/EzBQ21ddIF6jD
nl2ZhcvWHvS63utWwXW68xkDXsLvjiat22YScRKnkkNAIvbBY4rvV1KwahUPaMTS
zWywr6caHxlKp7McZ4MJVIqUAeZUn4KYgSksi2IsfPA7qi8WpSaKGsOZFBD79DJC
wqzdLLBzEtg6okzgC5nFtwIDAQABo1MwUTAdBgNVHQ4EFgQUgKAUBaaA1XD8KqGg
1XEr74W4lrkwHwYDVR0jBBgwFoAUgKAUBaaA1XD8KqGg1XEr74W4lrkwDwYDVR0T
AQH/BAUwAwEB/zANBgkqhkiG9w0BAQsFAAOCAQEArNBpG8oCDKX9ERbMgvgm3qWk
FKmx+h58aiZVoMwfBsf2njZ6BzRoEOvluMDPe+pt8hhST5yaOsMUYIqrn+s692I9
17JRfrFhCp+4GT8oe/ZnSNTPm2zOzm0VXFkfDF53YGzdGTWXLH+pJpw4drCNoBoA
yloyF/JJGJ2ZMbnwuDtsPbpjup8qHLiQYjxj4hUWyXU+nbytGK/i8z8HHc7acOpd
9+MXEcKwUkthXzG0M/0bzz4GwWZ6PPmbI5EEqFGBzMef58/mbHDigl9/o3kUlJtB
tcCZhP5KEu6XKKc1GcTqbyA0vi92YyyZViUa36hhVrNqPxtpclir+lcnNgnlqg==
-----END CERTIFICATE-----
");

        public const string PfxDataPassword = "12345";

        public static SecureString CreatePfxDataPasswordSecureString()
        {
            var s = new SecureString();

            // WARNING:
            // A key value of SecureString is in keeping string data off of the GC heap, such that it can
            // be reliably cleared when no longer needed.  Creating a SecureString from a string or converting
            // a SecureString to a string diminishes that value. These conversion functions are for testing that 
            // SecureString works, and does not represent a pattern to follow in any non-test situation.
            foreach (char c in PfxDataPassword.ToCharArray())
            {
                s.AppendChar(c);
            }

            return s;
        }

        public static readonly byte[] PfxSha1Empty_ExpectedSig = (
            "44b15120b8c7de19b4968d761600ffb8c54e5d0c1bcaba0880a20ab48912c8fd" + 
            "fa81b28134eabf58f3211a0d1eefdaae115e7872d5a67045c3b62a5da4393940" +
            "e5a496413a6d55ea6309d0013e90657c83c6e40aa8fafeee66acbb6661c14190" + 
            "11e1fde6f4fcc328bd7e537e4aa2dbe216d8f1f3aa7e5ec60eb9cfdca7a41d74").HexToByteArray();

        public static readonly byte[] PfxData = (
            "3082063A020103308205F606092A864886F70D010701A08205E7048205E33082" +
            "05DF3082035806092A864886F70D010701A08203490482034530820341308203" +
            "3D060B2A864886F70D010C0A0102A08202B6308202B2301C060A2A864886F70D" +
            "010C0103300E04085052002C7DA2C2A6020207D0048202907D485E3BFC6E6457" +
            "C811394C145D0E8A18325646854E4FF0097BC5A98547F5AD616C8EFDA8505AA8" +
            "7564ED4800A3139759497C60C6688B51F376ACAE906429C8771CB1428226B68A" +
            "6297207BCC9DD7F9563478DD83880AAB2304B545759B2275305DF4EFF9FAC24A" +
            "3CC9D3B2D672EFE45D8F48E24A16506C1D7566FC6D1B269FBF201B3AC3309D3E" +
            "BC6FD606257A7A707AA2F790EA3FE7A94A51138540C5319010CBA6DE9FB9D85F" +
            "CDC78DA60E33DF2F21C46FB9A8554B4F82E0A6EDBA4DB5585D77D331D35DAAED" +
            "51B6A5A3E000A299880FB799182C8CA3004B7837A9FEB8BFC76778089993F3D1" +
            "1D70233608AF7C50722D680623D2BF54BD4B1E7A604184D9F44E0AF8099FFA47" +
            "1E5536E7902793829DB9902DDB61264A62962950AD274EA516B2D44BE9036530" +
            "016E607B73F341AEEFED2211F6330364738B435B0D2ED6C57747F6C8230A053F" +
            "78C4DD65DB83B26C6A47836A6CBBAB92CBB262C6FB6D08632B4457F5FA8EABFA" +
            "65DB34157E1D301E9085CC443582CDD15404314872748545EB3FC3C574882655" +
            "8C9A85F966E315775BBE9DA34D1E8B6DADC3C9E120C6D6A2E1CFFE4EB014C3CE" +
            "FBC19356CE33DAC60F93D67A4DE247B0DAE13CD8B8C9F15604CC0EC9968E3AD7" +
            "F57C9F53C45E2ECB0A0945EC0BA04BAA15B48D8596EDC9F5FE9165A5D21949FB" +
            "5FE30A920AD2C0F78799F6443C300629B8CA4DCA19B9DBF1E27AAB7B12271228" +
            "119A95C9822BE6439414BEEAE24002B46EB97E030E18BD810ADE0BCF4213A355" +
            "038B56584B2FBCC3F5EA215D0CF667FFD823EA03AB62C3B193DFB4450AABB50B" +
            "AF306E8088EE7384FA2FDFF03E0DD7ACD61832223E806A94D46E196462522808" +
            "3163F1CAF333FDBBE2D54CA86968867CE0B6DD5E5B7F0633C6FAB4A19CC14F64" +
            "5EC14D0B1436F7623174301306092A864886F70D010915310604040100000030" +
            "5D06092B060104018237110131501E4E004D006900630072006F0073006F0066" +
            "00740020005300740072006F006E0067002000430072007900700074006F0067" +
            "007200610070006800690063002000500072006F007600690064006500723082" +
            "027F06092A864886F70D010706A08202703082026C0201003082026506092A86" +
            "4886F70D010701301C060A2A864886F70D010C0106300E0408E0C117E67A75D8" +
            "EB020207D080820238292882408B31826F0DC635F9BBE7C199A48A3B4FEFC729" +
            "DBF95508D6A7D04805A8DD612427F93124F522AC7D3C6F4DDB74D937F57823B5" +
            "B1E8CFAE4ECE4A1FFFD801558D77BA31985AA7F747D834CBE84464EF777718C9" +
            "865C819D6C9DAA0FA25E2A2A80B3F2AAA67D40E382EB084CCA85E314EA40C3EF" +
            "3ED1593904D7A16F37807C99AF06C917093F6C5AAEBB12A6C58C9956D4FBBDDE" +
            "1F1E389989C36E19DD38D4B978D6F47131E458AB68E237E40CB6A87F21C8773D" +
            "E845780B50995A51F041106F47C740B3BD946038984F1AC9E91230616480962F" +
            "11B0683F8802173C596C4BD554642F51A76F9DFFF9053DEF7B3C3F759FC7EEAC" +
            "3F2386106C4B8CB669589E004FB235F0357EA5CF0B5A6FC78A6D941A3AE44AF7" +
            "B601B59D15CD1EC61BCCC481FBB83EAE2F83153B41E71EF76A2814AB59347F11" +
            "6AB3E9C1621668A573013D34D13D3854E604286733C6BAD0F511D7F8FD6356F7" +
            "C3198D0CB771AF27F4B5A3C3B571FDD083FD68A9A1EEA783152C436F7513613A" +
            "7E399A1DA48D7E55DB7504DC47D1145DF8D7B6D32EAA4CCEE06F98BB3DDA2CC0" +
            "D0564A962F86DFB122E4F7E2ED6F1B509C58D4A3B2D0A68788F7E313AECFBDEF" +
            "456C31B96FC13586E02AEB65807ED83BB0CB7C28F157BC95C9C593C919469153" +
            "9AE3C620ED1D4D4AF0177F6B9483A5341D7B084BC5B425AFB658168EE2D8FB2B" +
            "FAB07A3BA061687A5ECD1F8DA9001DD3E7BE793923094ABB0F2CF4D24CB071B9" +
            "E568B18336BB4DC541352C9785C48D0F0E53066EB2009EFCB3E5644ED12252C1" +
            "BC303B301F300706052B0E03021A04144DEAB829B57A3156AEBC8239C0E7E884" +
            "EFD96E680414E147930B932899741C92D7652268938770254A2B020207D0").HexToByteArray();

        public static byte[] StoreSavedAsPfxData = (
            "3082070406092a864886f70d010702a08206f5308206f10201013100300b0609" +
            "2a864886f70d010701a08206d9308201e530820152a0030201020210d5b5bc1c" +
            "458a558845bff51cb4dff31c300906052b0e03021d05003011310f300d060355" +
            "040313064d794e616d65301e170d3130303430313038303030305a170d313130" +
            "3430313038303030305a3011310f300d060355040313064d794e616d6530819f" +
            "300d06092a864886f70d010101050003818d0030818902818100b11e30ea8742" +
            "4a371e30227e933ce6be0e65ff1c189d0d888ec8ff13aa7b42b68056128322b2" +
            "1f2b6976609b62b6bc4cf2e55ff5ae64e9b68c78a3c2dacc916a1bc7322dd353" +
            "b32898675cfb5b298b176d978b1f12313e3d865bc53465a11cca106870a4b5d5" +
            "0a2c410938240e92b64902baea23eb093d9599e9e372e48336730203010001a3" +
            "46304430420603551d01043b3039801024859ebf125e76af3f0d7979b4ac7a96" +
            "a1133011310f300d060355040313064d794e616d658210d5b5bc1c458a558845" +
            "bff51cb4dff31c300906052b0e03021d0500038181009bf6e2cf830ed485b86d" +
            "6b9e8dffdcd65efc7ec145cb9348923710666791fcfa3ab59d689ffd7234b787" +
            "2611c5c23e5e0714531abadb5de492d2c736e1c929e648a65cc9eb63cd84e57b" +
            "5909dd5ddf5dbbba4a6498b9ca225b6e368b94913bfc24de6b2bd9a26b192b95" +
            "7304b89531e902ffc91b54b237bb228be8afcda26476308204ec308203d4a003" +
            "020102021333000000b011af0a8bd03b9fdd0001000000b0300d06092a864886" +
            "f70d01010505003079310b300906035504061302555331133011060355040813" +
            "0a57617368696e67746f6e3110300e060355040713075265646d6f6e64311e30" +
            "1c060355040a13154d6963726f736f667420436f72706f726174696f6e312330" +
            "210603550403131a4d6963726f736f667420436f6465205369676e696e672050" +
            "4341301e170d3133303132343232333333395a170d3134303432343232333333" +
            "395a308183310b3009060355040613025553311330110603550408130a576173" +
            "68696e67746f6e3110300e060355040713075265646d6f6e64311e301c060355" +
            "040a13154d6963726f736f667420436f72706f726174696f6e310d300b060355" +
            "040b13044d4f5052311e301c060355040313154d6963726f736f667420436f72" +
            "706f726174696f6e30820122300d06092a864886f70d01010105000382010f00" +
            "3082010a0282010100e8af5ca2200df8287cbc057b7fadeeeb76ac28533f3adb" +
            "407db38e33e6573fa551153454a5cfb48ba93fa837e12d50ed35164eef4d7adb" +
            "137688b02cf0595ca9ebe1d72975e41b85279bf3f82d9e41362b0b40fbbe3bba" +
            "b95c759316524bca33c537b0f3eb7ea8f541155c08651d2137f02cba220b10b1" +
            "109d772285847c4fb91b90b0f5a3fe8bf40c9a4ea0f5c90a21e2aae3013647fd" +
            "2f826a8103f5a935dc94579dfb4bd40e82db388f12fee3d67a748864e162c425" +
            "2e2aae9d181f0e1eb6c2af24b40e50bcde1c935c49a679b5b6dbcef9707b2801" +
            "84b82a29cfbfa90505e1e00f714dfdad5c238329ebc7c54ac8e82784d37ec643" +
            "0b950005b14f6571c50203010001a38201603082015c30130603551d25040c30" +
            "0a06082b06010505070303301d0603551d0e041604145971a65a334dda980780" +
            "ff841ebe87f9723241f230510603551d11044a3048a4463044310d300b060355" +
            "040b13044d4f5052313330310603550405132a33313539352b34666166306237" +
            "312d616433372d346161332d613637312d373662633035323334346164301f06" +
            "03551d23041830168014cb11e8cad2b4165801c9372e331616b94c9a0a1f3056" +
            "0603551d1f044f304d304ba049a0478645687474703a2f2f63726c2e6d696372" +
            "6f736f66742e636f6d2f706b692f63726c2f70726f64756374732f4d6963436f" +
            "645369675043415f30382d33312d323031302e63726c305a06082b0601050507" +
            "0101044e304c304a06082b06010505073002863e687474703a2f2f7777772e6d" +
            "6963726f736f66742e636f6d2f706b692f63657274732f4d6963436f64536967" +
            "5043415f30382d33312d323031302e637274300d06092a864886f70d01010505" +
            "00038201010031d76e2a12573381d59dc6ebf93ad4444d089eee5edf6a5bb779" +
            "cf029cbc76689e90a19c0bc37fa28cf14dba9539fb0de0e19bf45d240f1b8d88" +
            "153a7cdbadceb3c96cba392c457d24115426300d0dff47ea0307e5e4665d2c7b" +
            "9d1da910fa1cb074f24f696b9ea92484daed96a0df73a4ef6a1aac4b629ef17c" +
            "c0147f48cd4db244f9f03c936d42d8e87ce617a09b68680928f90297ef1103ba" +
            "6752adc1e9b373a6d263cd4ae23ee4f34efdffa1e0bb02133b5d20de553fa3ae" +
            "9040313875285e04a9466de6f57a7940bd1fcde845d5aee25d3ef575c7e66663" +
            "60ccd59a84878d2430f7ef34d0631db142674a0e4bbf3a0eefb6953aa738e425" +
            "9208a68866823100").HexToByteArray();

        public static byte[] StoreSavedAsCerData = (
            "308201e530820152a0030201020210d5b5bc1c458a558845bff51cb4dff31c30" +
            "0906052b0e03021d05003011310f300d060355040313064d794e616d65301e17" +
            "0d3130303430313038303030305a170d3131303430313038303030305a301131" +
            "0f300d060355040313064d794e616d6530819f300d06092a864886f70d010101" +
            "050003818d0030818902818100b11e30ea87424a371e30227e933ce6be0e65ff" +
            "1c189d0d888ec8ff13aa7b42b68056128322b21f2b6976609b62b6bc4cf2e55f" +
            "f5ae64e9b68c78a3c2dacc916a1bc7322dd353b32898675cfb5b298b176d978b" +
            "1f12313e3d865bc53465a11cca106870a4b5d50a2c410938240e92b64902baea" +
            "23eb093d9599e9e372e48336730203010001a346304430420603551d01043b30" +
            "39801024859ebf125e76af3f0d7979b4ac7a96a1133011310f300d0603550403" +
            "13064d794e616d658210d5b5bc1c458a558845bff51cb4dff31c300906052b0e" +
            "03021d0500038181009bf6e2cf830ed485b86d6b9e8dffdcd65efc7ec145cb93" +
            "48923710666791fcfa3ab59d689ffd7234b7872611c5c23e5e0714531abadb5d" +
            "e492d2c736e1c929e648a65cc9eb63cd84e57b5909dd5ddf5dbbba4a6498b9ca" +
            "225b6e368b94913bfc24de6b2bd9a26b192b957304b89531e902ffc91b54b237" +
            "bb228be8afcda26476").HexToByteArray();

        public static byte[] StoreSavedAsSerializedCerData = (
            "0200000001000000bc0000001c0000006c000000010000000000000000000000" +
            "00000000020000007b00370037004500420044003000320044002d0044003800" +
            "440045002d0034003700350041002d0038003800360037002d00440032003000" +
            "4200300030003600340045003400390046007d00000000004d00690063007200" +
            "6f0073006f006600740020005300740072006f006e0067002000430072007900" +
            "700074006f0067007200610070006800690063002000500072006f0076006900" +
            "64006500720000002000000001000000e9010000308201e530820152a0030201" +
            "020210d5b5bc1c458a558845bff51cb4dff31c300906052b0e03021d05003011" +
            "310f300d060355040313064d794e616d65301e170d3130303430313038303030" +
            "305a170d3131303430313038303030305a3011310f300d060355040313064d79" +
            "4e616d6530819f300d06092a864886f70d010101050003818d00308189028181" +
            "00b11e30ea87424a371e30227e933ce6be0e65ff1c189d0d888ec8ff13aa7b42" +
            "b68056128322b21f2b6976609b62b6bc4cf2e55ff5ae64e9b68c78a3c2dacc91" +
            "6a1bc7322dd353b32898675cfb5b298b176d978b1f12313e3d865bc53465a11c" +
            "ca106870a4b5d50a2c410938240e92b64902baea23eb093d9599e9e372e48336" +
            "730203010001a346304430420603551d01043b3039801024859ebf125e76af3f" +
            "0d7979b4ac7a96a1133011310f300d060355040313064d794e616d658210d5b5" +
            "bc1c458a558845bff51cb4dff31c300906052b0e03021d0500038181009bf6e2" +
            "cf830ed485b86d6b9e8dffdcd65efc7ec145cb9348923710666791fcfa3ab59d" +
            "689ffd7234b7872611c5c23e5e0714531abadb5de492d2c736e1c929e648a65c" +
            "c9eb63cd84e57b5909dd5ddf5dbbba4a6498b9ca225b6e368b94913bfc24de6b" +
            "2bd9a26b192b957304b89531e902ffc91b54b237bb228be8afcda26476").HexToByteArray();

        public static byte[] StoreSavedAsSerializedStoreData = (
            "00000000434552540200000001000000bc0000001c0000006c00000001000000" +
            "000000000000000000000000020000007b003700370045004200440030003200" +
            "44002d0044003800440045002d0034003700350041002d003800380036003700" +
            "2d004400320030004200300030003600340045003400390046007d0000000000" +
            "4d006900630072006f0073006f006600740020005300740072006f006e006700" +
            "2000430072007900700074006f00670072006100700068006900630020005000" +
            "72006f007600690064006500720000002000000001000000e9010000308201e5" +
            "30820152a0030201020210d5b5bc1c458a558845bff51cb4dff31c300906052b" +
            "0e03021d05003011310f300d060355040313064d794e616d65301e170d313030" +
            "3430313038303030305a170d3131303430313038303030305a3011310f300d06" +
            "0355040313064d794e616d6530819f300d06092a864886f70d01010105000381" +
            "8d0030818902818100b11e30ea87424a371e30227e933ce6be0e65ff1c189d0d" +
            "888ec8ff13aa7b42b68056128322b21f2b6976609b62b6bc4cf2e55ff5ae64e9" +
            "b68c78a3c2dacc916a1bc7322dd353b32898675cfb5b298b176d978b1f12313e" +
            "3d865bc53465a11cca106870a4b5d50a2c410938240e92b64902baea23eb093d" +
            "9599e9e372e48336730203010001a346304430420603551d01043b3039801024" +
            "859ebf125e76af3f0d7979b4ac7a96a1133011310f300d060355040313064d79" +
            "4e616d658210d5b5bc1c458a558845bff51cb4dff31c300906052b0e03021d05" +
            "00038181009bf6e2cf830ed485b86d6b9e8dffdcd65efc7ec145cb9348923710" +
            "666791fcfa3ab59d689ffd7234b7872611c5c23e5e0714531abadb5de492d2c7" +
            "36e1c929e648a65cc9eb63cd84e57b5909dd5ddf5dbbba4a6498b9ca225b6e36" +
            "8b94913bfc24de6b2bd9a26b192b957304b89531e902ffc91b54b237bb228be8" +
            "afcda264762000000001000000f0040000308204ec308203d4a0030201020213" +
            "33000000b011af0a8bd03b9fdd0001000000b0300d06092a864886f70d010105" +
            "05003079310b3009060355040613025553311330110603550408130a57617368" +
            "696e67746f6e3110300e060355040713075265646d6f6e64311e301c06035504" +
            "0a13154d6963726f736f667420436f72706f726174696f6e3123302106035504" +
            "03131a4d6963726f736f667420436f6465205369676e696e6720504341301e17" +
            "0d3133303132343232333333395a170d3134303432343232333333395a308183" +
            "310b3009060355040613025553311330110603550408130a57617368696e6774" +
            "6f6e3110300e060355040713075265646d6f6e64311e301c060355040a13154d" +
            "6963726f736f667420436f72706f726174696f6e310d300b060355040b13044d" +
            "4f5052311e301c060355040313154d6963726f736f667420436f72706f726174" +
            "696f6e30820122300d06092a864886f70d01010105000382010f003082010a02" +
            "82010100e8af5ca2200df8287cbc057b7fadeeeb76ac28533f3adb407db38e33" +
            "e6573fa551153454a5cfb48ba93fa837e12d50ed35164eef4d7adb137688b02c" +
            "f0595ca9ebe1d72975e41b85279bf3f82d9e41362b0b40fbbe3bbab95c759316" +
            "524bca33c537b0f3eb7ea8f541155c08651d2137f02cba220b10b1109d772285" +
            "847c4fb91b90b0f5a3fe8bf40c9a4ea0f5c90a21e2aae3013647fd2f826a8103" +
            "f5a935dc94579dfb4bd40e82db388f12fee3d67a748864e162c4252e2aae9d18" +
            "1f0e1eb6c2af24b40e50bcde1c935c49a679b5b6dbcef9707b280184b82a29cf" +
            "bfa90505e1e00f714dfdad5c238329ebc7c54ac8e82784d37ec6430b950005b1" +
            "4f6571c50203010001a38201603082015c30130603551d25040c300a06082b06" +
            "010505070303301d0603551d0e041604145971a65a334dda980780ff841ebe87" +
            "f9723241f230510603551d11044a3048a4463044310d300b060355040b13044d" +
            "4f5052313330310603550405132a33313539352b34666166306237312d616433" +
            "372d346161332d613637312d373662633035323334346164301f0603551d2304" +
            "1830168014cb11e8cad2b4165801c9372e331616b94c9a0a1f30560603551d1f" +
            "044f304d304ba049a0478645687474703a2f2f63726c2e6d6963726f736f6674" +
            "2e636f6d2f706b692f63726c2f70726f64756374732f4d6963436f6453696750" +
            "43415f30382d33312d323031302e63726c305a06082b06010505070101044e30" +
            "4c304a06082b06010505073002863e687474703a2f2f7777772e6d6963726f73" +
            "6f66742e636f6d2f706b692f63657274732f4d6963436f645369675043415f30" +
            "382d33312d323031302e637274300d06092a864886f70d010105050003820101" +
            "0031d76e2a12573381d59dc6ebf93ad4444d089eee5edf6a5bb779cf029cbc76" +
            "689e90a19c0bc37fa28cf14dba9539fb0de0e19bf45d240f1b8d88153a7cdbad" +
            "ceb3c96cba392c457d24115426300d0dff47ea0307e5e4665d2c7b9d1da910fa" +
            "1cb074f24f696b9ea92484daed96a0df73a4ef6a1aac4b629ef17cc0147f48cd" +
            "4db244f9f03c936d42d8e87ce617a09b68680928f90297ef1103ba6752adc1e9" +
            "b373a6d263cd4ae23ee4f34efdffa1e0bb02133b5d20de553fa3ae9040313875" +
            "285e04a9466de6f57a7940bd1fcde845d5aee25d3ef575c7e6666360ccd59a84" +
            "878d2430f7ef34d0631db142674a0e4bbf3a0eefb6953aa738e4259208a68866" +
            "82000000000000000000000000").HexToByteArray();

        public static byte[] DssCer = (
            "3082025d3082021da00302010202101e9ae1e91e07de8640ac7af21ac22e8030" +
            "0906072a8648ce380403300e310c300a06035504031303466f6f301e170d3135" +
            "303232343232313734375a170d3136303232343232313734375a300e310c300a" +
            "06035504031303466f6f308201b73082012c06072a8648ce3804013082011f02" +
            "818100871018cc42552d14a5a9286af283f3cfba959b8835ec2180511d0dceb8" +
            "b979285708c800fc10cb15337a4ac1a48ed31394072015a7a6b525986b49e5e1" +
            "139737a794833c1aa1e0eaaa7e9d4efeb1e37a65dbc79f51269ba41e8f0763aa" +
            "613e29c81c3b977aeeb3d3c3f6feb25c270cdcb6aee8cd205928dfb33c44d2f2" +
            "dbe819021500e241edcf37c1c0e20aadb7b4e8ff7aa8fde4e75d02818100859b" +
            "5aeb351cf8ad3fabac22ae0350148fd1d55128472691709ec08481584413e9e5" +
            "e2f61345043b05d3519d88c021582ccef808af8f4b15bd901a310fefd518af90" +
            "aba6f85f6563db47ae214a84d0b7740c9394aa8e3c7bfef1beedd0dafda079bf" +
            "75b2ae4edb7480c18b9cdfa22e68a06c0685785f5cfb09c2b80b1d05431d0381" +
            "8400028180089a43f439b924bef3529d8d6206d1fca56a55caf52b41d6ce371e" +
            "bf07bda132c8eadc040007fcf4da06c1f30504ebd8a77d301f5a4702f01f0d2a" +
            "0707ac1da38dd3251883286e12456234da62eda0df5fe2fa07cd5b16f3638bec" +
            "ca7786312da7d3594a4bb14e353884da0e9aecb86e3c9bdb66fca78ea85e1cc3" +
            "f2f8bf0963300906072a8648ce380403032f00302c021461f6d143a47a4f7e0e" +
            "0ef9848b7f83eacbf83ffd021420e2ac47e656874633e01b0d207a99280c1127" +
            "01").HexToByteArray();

        public static byte[] Dsa1024Pfx = (
            "308206EE020103308206B406092A864886F70D010701A08206A5048206A13082" +
            "069D3082043706092A864886F70D010706A0820428308204240201003082041D" +
            "06092A864886F70D010701301C060A2A864886F70D010C0106300E04084AF212" +
            "89D5D7E2E702020800808203F0DECCF218AC91F26BAB026998AB77C7629D20DB" +
            "E2FB7022A3C4A1CECD743C0F932E944AE229DAFB61AD76C4DEB6995DF4F4BA01" +
            "2DBAD5C63A4C846E0807FCA0BC4A162CDFBAB4B3C4D304F473B3ACC1D268436E" +
            "F537DAE97ECC3C634C8DF2A294CC23E904A169F369021A0C024A03DE98A65B0F" +
            "3F14D6910525D76AD98B91E67BB7398E245CF48A4D2A5603CFCCF4E547D7EDAB" +
            "669D9A8597C6839119EB9FD932D1E4BA8B45D3317186CDA2EFF247BCFD64A5CA" +
            "ED604BF7033E423CC21CEC6454FE3B74E03A26C51A1C3519CE339FBE9F10B81D" +
            "DF6A0AAB4F8166D90B6F52B3439AB4B5273D0A506E3E01869F8FEBD1521EF8E5" +
            "BFB357FA630E3C988926EF3ACC0A0F4176FE8A93337C1A5C6DEAB5758EC2F07C" +
            "11E8B2495ECDE58D12312CCCA2E8B2EE8564B533D18C7A26A9290394C2A9942C" +
            "295EBB0317F5695103627519567960908323FFE6560AD054C97800218A52F37A" +
            "DDE4E7F18EF3BF3718A9D7BF57B700DBEB5AB86598C9604A4546995E34DBABBB" +
            "6A9FB483A3C2DFE6046DFD54F2D7AC61C062AF04B7FBAC395C5DD19408D6926A" +
            "93B896BFB92DA6F7F5A4E54EDBE2CFBB56576878150676ADB0D37E0177B91E0D" +
            "F09D7B37769E66842DD40C7B1422127F152A165BC9669168885BA0243C9641B4" +
            "48F68575AA6AB9247A49A61AC3C683EE057B7676B9610CF9100096FC46BDC8B9" +
            "BAA03535815D5E98BA3ABC1E18E39B50A8AF8D81E30F2DFD6AF5D0F9FC3636AB" +
            "69E128C793571723A79E42FC7C1BD7F39BD45FBE9C39EEB010005435BEC19844" +
            "2058033D2601B83124BD369DADB831317E0B2C28CE7535A2E89D8A0E5E34E252" +
            "3B0FCEC34FF26A2B80566F4D86F958F70106BF3322FA70A3312E48EAA130246A" +
            "07412E93FDE91F633F758BC49311F6CBBAEC5D2F22AFCD696F72BC22E7DE6C00" +
            "3275DFEC47E3848226FE9DBA184EA711E051B267C584749F897EFE7EAFD02F1D" +
            "BF3FD8E882474CA1F45509EF2E7B82F35B677CB88ED42AF729848EE2B424B0CE" +
            "2E9AAC945BABA550C20D5B25075A30FE70D8CAA5A527A35F1DF17BCCB91930C1" +
            "7120C625667120E0806C2B51EDFF540F928BD555FB48DBCB83CCCE0C385E78C8" +
            "65BE715AE6F8BE472E5FC187EBE3FEFD8D7FE62D4DB2EE61F42D24D81FAA9179" +
            "0FB17E8EBC8E219B6F9E039F5AB3BC4870821D474B36C8F8D0583D9DC06E4383" +
            "D03424420B8C8B26276877166A0F51E22F0D8FA60A070CFBD47EAFBC717C879C" +
            "B5A1EA69C4C2A38F26A1EEF96A0C32BFCECCE4EA97E90A425066B1DD0891353F" +
            "766EB9F2BFA2563A815DAF3639EBB147E1E8757A6BFAB902C4A8F037AD47E03F" +
            "AF2E019FCF6CA7430BDFEA4B45B28ED746BB90E09BEF7B370A75E7924BBA0920" +
            "25FE654A9A197A5B8BBBE43DC7C892FF14E75A37EB97FC489AB121A43E308202" +
            "5E06092A864886F70D010701A082024F0482024B3082024730820243060B2A86" +
            "4886F70D010C0A0102A082017630820172301C060A2A864886F70D010C010330" +
            "0E0408ECB4D1550DA52C6302020800048201509322DC0193DD9E79ADAFD38827" +
            "AD6DE9299327DDDF6E9DF4FB70D53A64951E4B814E90D2A19B3F4B8E39A2F851" +
            "A3E5E9B9EB947DD248A3E5F5EB458F3323D4656709E97C6BD59238C4D1F26AB6" +
            "7D73235FAE7780D98705957B6650AC0DE3E2D46E22455D0A105D138F16A84839" +
            "14EDDF5C518B748558704ED3AE4A8C4914F667BBDE07978E4A4FC66194F6B86B" +
            "AB9F558EDE890C25DFB97C59653906CC573B5DEB62165CFF8A5F4F8059A478EB" +
            "F6FED75F1DACDC612C2E271E25A7083E15D33697270FD442D79FFCB25DB135F9" +
            "8E580DC9CE14F73C3B847931AF821C77718455F595CA15B86386F3FCC5962262" +
            "5FC916DDB4A08479DCB49FF7444333FA99FBB22F1AEC1876CF1E099F7A4ECA85" +
            "A325A8623E071EEA9359194EEE712F73076C5EB72AA243D0C0978B934BC8596F" +
            "8353FD3CA859EEA457C6175E82AE5854CC7B6598A1E980332F56AB1EE1208277" +
            "4A91A63181B9302306092A864886F70D01091531160414E6335FA7097AB6DE4A" +
            "1CDB0C678D7A929883FB6430819106092B06010401823711013181831E818000" +
            "4D006900630072006F0073006F0066007400200045006E00680061006E006300" +
            "650064002000440053005300200061006E006400200044006900660066006900" +
            "65002D00480065006C006C006D0061006E002000430072007900700074006F00" +
            "67007200610070006800690063002000500072006F0076006900640065007230" +
            "313021300906052B0E03021A0500041466FD3518CEBBD69877BA663C9E8D7092" +
            "8E8A98F30408DFB5AE610308BCF802020800").HexToByteArray();

        public const string Dsa1024PfxPassword = "1234";

        public static byte[] Dsa1024Cert = (
            "3082038D3082034AA003020102020900AB740A714AA83C92300B060960864801" +
            "650304030230818D310B3009060355040613025553311330110603550408130A" +
            "57617368696E67746F6E3110300E060355040713075265646D6F6E64311E301C" +
            "060355040A13154D6963726F736F667420436F72706F726174696F6E3120301E" +
            "060355040B13172E4E4554204672616D65776F726B2028436F72654658293115" +
            "30130603550403130C313032342D62697420445341301E170D31353131323531" +
            "34343030335A170D3135313232353134343030335A30818D310B300906035504" +
            "0613025553311330110603550408130A57617368696E67746F6E3110300E0603" +
            "55040713075265646D6F6E64311E301C060355040A13154D6963726F736F6674" +
            "20436F72706F726174696F6E3120301E060355040B13172E4E4554204672616D" +
            "65776F726B2028436F7265465829311530130603550403130C313032342D6269" +
            "7420445341308201B73082012C06072A8648CE3804013082011F02818100AEE3" +
            "309FC7C9DB750D4C3797D333B3B9B234B462868DB6FFBDED790B7FC8DDD574C2" +
            "BD6F5E749622507AB2C09DF5EAAD84859FC0706A70BB8C9C8BE22B4890EF2325" +
            "280E3A7F9A3CE341DBABEF6058D063EA6783478FF8B3B7A45E0CA3F7BAC9995D" +
            "CFDDD56DF168E91349130F719A4E717351FAAD1A77EAC043611DC5CC5A7F0215" +
            "00D23428A76743EA3B49C62EF0AA17314A85415F0902818100853F830BDAA738" +
            "465300CFEE02418E6B07965658EAFDA7E338A2EB1531C0E0CA5EF1A12D9DDC7B" +
            "550A5A205D1FF87F69500A4E4AF5759F3F6E7F0C48C55396B738164D9E35FB50" +
            "6BD50E090F6A497C70E7E868C61BD4477C1D62922B3DBB40B688DE7C175447E2" +
            "E826901A109FAD624F1481B276BF63A665D99C87CEE9FD063303818400028180" +
            "25B8E7078E149BAC352667623620029F5E4A5D4126E336D56F1189F9FF71EA67" +
            "1B844EBD351514F27B69685DDF716B32F102D60EA520D56F544D19B2F08F5D9B" +
            "DDA3CBA3A73287E21E559E6A07586194AFAC4F6E721EDCE49DE0029627626D7B" +
            "D30EEB337311DB4FF62D7608997B6CC32E9C42859820CA7EF399590D5A388C48" +
            "A330302E302C0603551D110425302387047F0000018710000000000000000000" +
            "0000000000000182096C6F63616C686F7374300B060960864801650304030203" +
            "3000302D021500B9316CC7E05C9F79197E0B41F6FD4E3FCEB72A8A0214075505" +
            "CCAECB18B7EF4C00F9C069FA3BC78014DE").HexToByteArray();

        public static byte[] CertWithPolicies = (
            "308201f33082015ca0030201020210134fb7082cf69bbb4930bfc8e1ca446130" +
            "0d06092a864886f70d0101050500300e310c300a06035504031303466f6f301e" +
            "170d3135303330313232343735385a170d3136303330313034343735385a300e" +
            "310c300a06035504031303466f6f30819f300d06092a864886f70d0101010500" +
            "03818d0030818902818100c252d52fb96658ddbb7d19dd9caaf203ec0376f77c" +
            "3012bd93e14bb22a6ff2b5ce8060a197e3fd8289fbff826746baae0db8d68b47" +
            "a1cf13678717d7db9a16dab028927173a3e843b3a7df8c5a4ff675957ea20703" +
            "6389a60a83d643108bd1293e2135a672a1cff10b7d5b3c78ab44d35e20ca6a5c" +
            "5b6f714c5bfd66ed4307070203010001a3523050301b06092b06010401823714" +
            "02040e1e0c00480065006c006c006f0000301a06092b0601040182371507040d" +
            "300b060357080902010302010230150603551d20040e300c3004060262133004" +
            "06027021300d06092a864886f70d0101050500038181001be04e59fbea63acfb" +
            "c8b6fd3d02dd7442532344cfbc124e924c0bacf23865e4ce2f442ad60ae457d8" +
            "4f7a1f05d50fb867c20e778e412a25237054555669ced01c1ce1ba8e8e57510f" +
            "73e1167c920f78aa5415dc5281f0c761fb25bb1ebc707bc003dd90911e649915" +
            "918cfe4f3176972f8afdc1cccd9705e7fb307a0c17d273").HexToByteArray();

        public static byte[] CertWithTemplateData = (
            "308201dc30820145a00302010202105101b8242daf6cae4c53bac68a948b0130" +
            "0d06092a864886f70d0101050500300e310c300a06035504031303466f6f301e" +
            "170d3135303330313232333133395a170d3136303330313034333133395a300e" +
            "310c300a06035504031303466f6f30819f300d06092a864886f70d0101010500" +
            "03818d0030818902818100a6dcff50bd1fe420301fea5fa56be93a7a53f2599c" +
            "e453cf3422bec797bac0ed78a03090a3754569e6494bcd585ac16a5ea5086344" +
            "3f25521085ca09580579cf0b46bd6e50015319fba5d2bd3724c53b20cdddf604" +
            "74bd7ef426aead9ca5ffea275a4b2b1b6f87c203ab8783559b75e319722886fb" +
            "eb784f5f06823906b2a9950203010001a33b3039301b06092b06010401823714" +
            "02040e1e0c00480065006c006c006f0000301a06092b0601040182371507040d" +
            "300b0603570809020103020102300d06092a864886f70d010105050003818100" +
            "962594da079523c26e2d3fc573fd17189ca33bedbeb2c38c92508fc2a865973b" +
            "e85ba686f765101aea0a0391b22fcfa6c0760eece91a0eb75501bf6871553f8d" +
            "6b089cf2ea63c872e0b4a178795b71826c4569857b45994977895e506dfb8075" +
            "ed1b1096987f2c8f65f2d6bbc788b1847b6ba13bee17ef6cb9c6a3392e13003f").HexToByteArray();

        public static byte[] ComplexNameInfoCert = (
            "308204BE30820427A00302010202080123456789ABCDEF300D06092A864886F70" +
            "D01010505003081A43110300E06035504061307436F756E747279310E300C0603" +
            "550408130553746174653111300F060355040713084C6F63616C6974793111300" +
            "F060355040A13084578616D706C654F31123010060355040B13094578616D706C" +
            "654F55311E301C06035504031315636E2E6973737565722E6578616D706C652E6" +
            "F72673126302406092A864886F70D0109011617697373756572656D61696C4065" +
            "78616D706C652E6F7267301E170D3133313131323134313531365A170D3134313" +
            "231333135313631375A3081A63110300E06035504061307436F756E747279310E" +
            "300C0603550408130553746174653111300F060355040713084C6F63616C69747" +
            "93111300F060355040A13084578616D706C654F31123010060355040B13094578" +
            "616D706C654F55311F301D06035504031316636E2E7375626A6563742E6578616" +
            "D706C652E6F72673127302506092A864886F70D01090116187375626A65637465" +
            "6D61696C406578616D706C652E6F7267305C300D06092A864886F70D010101050" +
            "0034B003048024100DC6FBBDA0300520DFBC9F046CC865D8876AEAC353807EA84" +
            "F58F92FE45EE03C22E970CAF41031D47F97C8A5117C62718482911A8A31B58D92" +
            "328BA3CF9E605230203010001A382023730820233300B0603551D0F0404030200" +
            "B0301D0603551D250416301406082B0601050507030106082B060105050703023" +
            "081FD0603551D120481F53081F28217646E73312E6973737565722E6578616D70" +
            "6C652E6F72678217646E73322E6973737565722E6578616D706C652E6F7267811" +
            "569616E656D61696C31406578616D706C652E6F7267811569616E656D61696C32" +
            "406578616D706C652E6F7267A026060A2B060104018237140203A0180C1669737" +
            "375657275706E31406578616D706C652E6F7267A026060A2B0601040182371402" +
            "03A0180C1669737375657275706E32406578616D706C652E6F7267861F6874747" +
            "03A2F2F757269312E6973737565722E6578616D706C652E6F72672F861F687474" +
            "703A2F2F757269322E6973737565722E6578616D706C652E6F72672F308201030" +
            "603551D110481FB3081F88218646E73312E7375626A6563742E6578616D706C65" +
            "2E6F72678218646E73322E7375626A6563742E6578616D706C652E6F726781157" +
            "3616E656D61696C31406578616D706C652E6F7267811573616E656D61696C3240" +
            "6578616D706C652E6F7267A027060A2B060104018237140203A0190C177375626" +
            "A65637475706E31406578616D706C652E6F7267A027060A2B0601040182371402" +
            "03A0190C177375626A65637475706E32406578616D706C652E6F7267862068747" +
            "4703A2F2F757269312E7375626A6563742E6578616D706C652E6F72672F862068" +
            "7474703A2F2F757269322E7375626A6563742E6578616D706C652E6F72672F300" +
            "D06092A864886F70D0101050500038181005CD44A247FF4DFBF2246CC04D7D57C" +
            "EF2B6D3A4BC83FF685F6B5196B65AFC8F992BE19B688E53E353EEA8B63951EC40" +
            "29008DE8B851E2C30B6BF73F219BCE651E5972E62D651BA171D1DA9831A449D99" +
            "AF4E2F4B9EE3FD0991EF305ADDA633C44EB5E4979751280B3F54F9CCD561AC27D" +
            "3426BC6FF32E8E1AAF9F7C0150A726B").HexToByteArray();

        internal static readonly byte[] MultiPrivateKeyPfx = (
            "30820F1602010330820ED606092A864886F70D010701A0820EC704820EC33082" +
            "0EBF308206A806092A864886F70D010701A08206990482069530820691308203" +
            "4C060B2A864886F70D010C0A0102A08202B6308202B2301C060A2A864886F70D" +
            "010C0103300E0408ED42EEFCD77BB2EB020207D00482029048F341D409492D23" +
            "D89C0C01DEE7EFFB6715B15D2BB558E9045D635CADFFFEC85C10A4849AB0657D" +
            "A17FE7EC578F779BA2DC129FA959664DC7E85DFD13CAC673E487208FE457223A" +
            "75732915FFCF3FF70F557B0846D62AD507300EA1770EDED82F7D8E6E75075728" +
            "A29D3BF829E75F09EF283A9DDEDDFBABC2E25698DA8C24E4FE34CD43C87554BF" +
            "55B1D4B2B0979F399AEC95B781C62CBE9E412329F9A9BCABF20F716A95F1D795" +
            "7C379A27587F6BBFA44A0B75FAAC15CA3730629C55E87990EE521BC4657EE2A4" +
            "41AF099A226D31707685A89A28EB27CA65512B70DEC09231369AA1A265D4F5C3" +
            "C5D17CB11DB54C70AB83EA28F4740D1F79D490F46F926FB267D5F0E4B2FE096D" +
            "F161A4FF9E9AC068EFCA999B3ED0A3BD05D8D1E3B67CF51E6A478154B427D87D" +
            "C861D0FE2A7A42600483D7B979DC71E8A00D0E805E3BB86E8673234DC1D14987" +
            "99272754A5FD5FEC118CF1E2B2A539B604FED5486A4E4D73FAAFF69023263B84" +
            "6870D6B8DB01E31CB3A1E4BA3588C1FA81C786745A33B95573D5381AB307827A" +
            "549A36AF535FD05E1247BB92C6C6FCB0E76E87F2E4C8136F37C9C19BE3001F59" +
            "FC5CB459C620B8E73711BF102D78F665F40E4D1A341370BC1FB7A5567C29359C" +
            "FFB938237002904BE59F5605AF96E8A670E2248AB71D27FE63E327077144F095" +
            "4CA815E0284E2FF5E1A11B2946276A99B91BF138A79B057436798AF72FD86842" +
            "881C5A5ECDA8A961A21553CC930703047F1F45699CEFEF26AAB6B7DBC65C8C62" +
            "4CA3286094596F2AA48268B9F5411058613185507332833AFB312D5780CEFF96" +
            "6DD05A2CB6E1B252D9656D8E92E63E6C0360F119232E954E11DE777D2DE1C208" +
            "F704DDB16E1351F49B42A859E3B6B2D94E1E2B3CD97F06B1123E9CCA049201E6" +
            "DB7273C0BDE63CC9318182301306092A864886F70D0109153106040401000000" +
            "306B06092B0601040182371101315E1E5C004D006900630072006F0073006F00" +
            "66007400200045006E00680061006E0063006500640020004300720079007000" +
            "74006F0067007200610070006800690063002000500072006F00760069006400" +
            "650072002000760031002E00303082033D060B2A864886F70D010C0A0102A082" +
            "02B6308202B2301C060A2A864886F70D010C0103300E04081F85B7ED57F6F934" +
            "020207D00482029051A5ADA683AAE06A699761CCF05CB081A4398A7B1256A250" +
            "84DBE1115BFAB07A5A9146BC22F2E4223FF25BCA1836AE218691815F20A27A1B" +
            "98D1FC78F84AFA7E90A55954EE5BEA47FFA35928A990CB47346767F6F4212DBC" +
            "D03FFF1E4D137979006B46B19A9FC3BC9B5036ED6F8582E2007D08DB94B2B576" +
            "E154719CAC90DFB6F238CA875FCBEBCF9E9F933E4451E6A2B60C2A0A8A35B5FD" +
            "20E5DDA000008DCCE95BBDF604A8F93001F594E402FF8649A6582DE5901EDF9D" +
            "ED7D6F9657C5A184D82690EFCFB2F25BFCE02BC56F0FF00595996EBF1BA25475" +
            "AB613461280DD641186237D8A3AB257BD6FB1BDC3768B00719D233E0D5FD26D0" +
            "8BA6EAB29D732B990FB9423E643E4663ABBA0D8885DD2A276EE02C92778261C7" +
            "853F708E2B9AF8D2E96416F676D0191BD24D0C8430BD419049F43C8E2A0C32F8" +
            "62207B3DA661577CE5933460D0EF69FAD7323098B55FEF3A9955FE632FBCE845" +
            "2BB5F3430AE2A9021EBF756CC7FDFC3E63581C8B0D7AB77760F447F868B59236" +
            "14DAA9C36AEBC67DC854B93C38E8A6D3AC11B1EE1D02855CE96ADEB840B626BF" +
            "C4B3BFD6487C9073F8A15F55BA945D58AD1636A7AED476EBDB5227A71144BF87" +
            "45192EF5CD177818F61836717ED9EB0A83BEEE582ADEDD407035E453083B17E7" +
            "C237009D9F04F355CEAB0C0E9AD6F13A3B54459FA05B19E02275FE2588258B63" +
            "A125F549D1B44C827CDC94260A02F4A1B42A30E675B9760D876685D6CA05C258" +
            "03BDE1F33D325CF6020A662B0F5DCCC8D77B941B273AC462F0D3E050CEB5AEF7" +
            "107C45372F7063EF1AB420CA555A6C9BE6E1067966755584346CDDE7C05B6132" +
            "E553B11C374DB90B54E5C096062349A1F6CB78A1A2D995C483541750CFA956DE" +
            "A0EB3667DE7AD78931C65B6E039B5DE461810B68C344D2723174301306092A86" +
            "4886F70D0109153106040402000000305D06092B060104018237110131501E4E" +
            "004D006900630072006F0073006F006600740020005300740072006F006E0067" +
            "002000430072007900700074006F006700720061007000680069006300200050" +
            "0072006F007600690064006500723082080F06092A864886F70D010706A08208" +
            "00308207FC020100308207F506092A864886F70D010701301C060A2A864886F7" +
            "0D010C0106300E04089ADEE71816BCD023020207D0808207C851AA1EA533FECA" +
            "BB26D3846FAEE8DEDB919C29F8B98BBBF785BC306C12A8ACB1437786C4689161" +
            "683718BB7E40EB60D9BE0C87056B5ECF20ACCB8BF7F36033B8FCB84ED1474E97" +
            "DE0A8709B563B6CF8E69DF4B3F970C92324946723C32D08B7C3A76C871C6B6C8" +
            "C56F2D3C4C00B8A809E65A4EB5EFECC011E2B10F0E44ECDA07B325417B249240" +
            "80844F6D7F1F6E420346EA85825EB830C7E05A5383412A9502A51F1AC07F315A" +
            "DE357F1F9FB2E6427976E78B8FF9CD6C2F9841F2D84658AC8747694EFD0C451B" +
            "7AC5B83D5F0780808417501666BB452B53CEB0698162D94541DE181A7968DB13" +
            "9F17A1076EDEB70B38B8881DBC6DE2B694070A5A1AA71E4CDFBF7F4D5DBCF166" +
            "46768364D3C74FA212E40CBE3BE7C51A74D271164D00E89F997FD418C51A7C2D" +
            "73130D7C6FCAA2CA65082CE38BFB753BB30CC71656529E8DBA4C4D0B7E1A79CF" +
            "2A052FFEFA2DEE3373115472AFD1F40A80B23AA6141D5CDE0A378FE6210D4EE6" +
            "9B8771D3E192FD989AEC14C26EA4845D261B8A45ABC1C8FA305449DCDEDA9882" +
            "DD4DDC69B2DE315645FBC3EE52090907E7687A22A63F538E030AB5A5413CA415" +
            "F1D70E70CB567261FB892A8B3BAFC72D632CD2FDCC0559E01D5C246CC27C9348" +
            "63CCFA52490E1F01D8D2D0AF2587E4D04011140A494FFA3CA42C5F645B94EE30" +
            "100DE019B27F66FFC035E49A65B2A3F6CB14EB1E2FFF1F25B5C87481BD8506F3" +
            "07E0B042A2C85B99ECA520B4AAC7DFF2B11C1213E4128A01765DDB27B867336B" +
            "8CCF148CE738465D46E7A0BEA466CD8BBCCE2E11B16E0F9D24FF2F2D7C9F8527" +
            "79ADBB818F87E4AFF7C21A9C2BC20D38209322A34B0B393B187C96583D3D73D9" +
            "440F994B2F320D3274848AB7167942179CFF725C2C7556CCC289A5E788C5B863" +
            "E6FCDD5E4B87E41458BEB3F43D14C7E5196C38CA36322F8B83064862178D5892" +
            "5AEF34F444A31A4FB18431D7D37C65ED519643BC7BD025F801390430022253AA" +
            "FCEA670726512C3532EA9F410DB8AA6628CC455E4AB3F478A6981DB9180B7A2A" +
            "24B365F37554CE04B08F22B3539D98BF9A1AC623BBF9A08DBEC951E9730C1318" +
            "02B2C40750AAE6A791B3219A96A5BAC7AE17A2F7EA02FF66D6FB36C2E6B6AB90" +
            "D821A6322BF3E8D82969756A474551DB9EAA8C587FC878F996F5FA1E1C39E983" +
            "F164B0A67897EB3755C378807FFDFE964C5C0F290784A08E8C925E85775A9B89" +
            "2E278F68C3C1DE72622AC10EA56D88C909EF4AC9F47ED61376737C1E43DBF0F8" +
            "9337F0684FA0B96E7A993EC328A6A5FBCDCB809ACBFDAE4ECE192A45480104ED" +
            "12820238AB6AC9C88CC9A82585FD29A81A7BC5BC591738A4D49A86D06B4E18BD" +
            "C83DFFAA60D8A0D4F70CC63D4E83812CB6753F3744545592D04223793E5B3051" +
            "25AAD8807A753D235769BD0280E2DE808B0CEE2B98B0F5562FF9EF68161A6B7E" +
            "08C8B105766EBCFC44AC858B1A89E34C099B194A8B24D1DBABC13909EFAF5B9A" +
            "9E77AEAF7DD9BE772FA01AB9518EB8864AE6D07D7DD7451797541D2F723BC71A" +
            "9C14ED1D811594E2C4A57017D4CB90FD82C195FA9B823DF1E2FFD965E3139F9A" +
            "6E8AAC36FA39CFA4C52E85D2A661F9D0D466720C5AB7ECDE968FF51B535B019A" +
            "3E9C76058E6F673A49CDD89EA7EC998BDADE71186EA084020A897A328753B72E" +
            "213A9D82443F7E34D94508199A2A63E71A12BD441C132201E9A3829B2727F23E" +
            "65C519F4DA2C40162A3A501B1BD57568ED75447FEAF8B42988CE25407644BFA0" +
            "B76059D275EC994BB336055E271751B32233D79A6E5E3AA700F3803CCA50586D" +
            "28934E3D4135FA043AF7DFAB977477283602B1739C4AF40E3856E75C34EB98C6" +
            "9A928ADE05B67A679630EFA14E64B2957EDD1AB4EC0B0E7BC38D4851EBF67928" +
            "33EACB62FB6C862B089E3066AE5EAAFD2A8B7FC712DE9BD2F488222EEB1FB91B" +
            "4E57C2D24092818965621C123280453EDCFA2EC9D9B50AFA437D1ED09EC36FD2" +
            "32B169ED301E0DB0BABE562B67130F90EBC85D325A90931A5B5A94736A4B3AAD" +
            "B8CA295F59AF7FF08CCFADE5AFBBC2346BC6D78D9E5F470E9BDFF547F2574B10" +
            "A48DD9D56B5B03E9E24D65C367B6E342A26A344111A66B1908EDAECD0834930D" +
            "A74E1CFE2E4B0636A7C18E51A27AD21992A2DCF466BAACAC227B90B5E61BED79" +
            "9C97DEE7EDB33CCAF5DAD7AAD3CACCDE59478CF69AE64B9065FCB436E1993514" +
            "C42872DD486ABB75A07A4ED46CDF0E12C0D73FAB83564CF1A814791971EC9C7C" +
            "6A08A13CE0453C2C3236C8B2E146D242E3D37A3ECF6C350D0B2AB956CB21057F" +
            "DC630750A71C61C66DE3D4A6DB187BEE2F86DEB93E723C5943EA17E699E93555" +
            "756920416BD6B267A4CFAC4EE90E96A6419302B4C0A3B9705509CA09EE92F184" +
            "FD2817BA09BE29E465909DB6C93E3C1CAF6DC29E1A5838F3C32CCB220235EF82" +
            "9CD21D1B3E960518A80D08AE7FF08D3AFB7451C823E9B8D49DAF66F503E4AE53" +
            "99FECFC958429D758C06EFF8338BC02457F6FE5053AA3C2F27D360058FD93566" +
            "3B55F026B504E39D86E7CE15F04B1C62BBFA0B1CA5E64FF0BD088D94FB1518E0" +
            "5B2F40BF9D71C61FC43E3AF8440570C44030F59D14B8858B7B8506B136E7E39B" +
            "B04F9AFEAF2FA292D28A8822046CEFDE381F2399370BDE9B97BC700418585C31" +
            "E9C353635ADAA6A00A833899D0EDA8F5FFC558D822AEB99C7E35526F5297F333" +
            "F9E758D4CD53277316608B1F7DB6AC71309A8542A356D407531BA1D3071BA9DC" +
            "02AE91C7DF2561AEBC3845A118B00D21913B4A401DDDC40CE983178EF26C4A41" +
            "343037301F300706052B0E03021A041454F0864331D9415EBA750C62FA93C97D" +
            "3402E1A40414B610EC75D16EA23BF253AAD061FAC376E1EAF684").HexToByteArray();

        internal static readonly byte[] EmptyPfx = (
            "304F020103301106092A864886F70D010701A004040230003037301F30070605" +
            "2B0E03021A0414822078BC83E955E314BDA908D76D4C5177CC94EB0414711018" +
            "F2897A44A90E92779CB655EA11814EC598").HexToByteArray();

        internal const string ChainPfxPassword = "test";

        internal static readonly byte[] ChainPfxBytes = (
            "308213790201033082133506092A864886F70D010701A0821326048213223082" +
            "131E3082036706092A864886F70D010701A08203580482035430820350308203" +
            "4C060B2A864886F70D010C0A0102A08202B6308202B2301C060A2A864886F70D" +
            "010C0103300E040811E8B9808BA6E96C020207D004820290D11DA8713602105C" +
            "95792D65BCDFC1B7E3708483BF6CD83008082F89DAE4D003F86081B153BD4D4A" +
            "C122E802752DEA29F07D0B7E8F0FB8A762B4CAA63360F9F72CA5846771980A6F" +
            "AE2643CD412E6E4A101625371BBD48CC6E2D25191D256B531B06DB7CDAC04DF3" +
            "E10C6DC556D5FE907ABF32F2966A561C988A544C19B46DF1BE531906F2CC2263" +
            "A301302A857075C7A9C48A395241925C6A369B60D176419D75E320008D5EFD91" +
            "5257B160F6CD643953E85F19EBE4E4F72B9B787CF93E95F819D1E43EF01CCFA7" +
            "48F0E7260734EA9BC6039BA7557BE6328C0149718A1D9ECF3355082DE697B6CD" +
            "630A9C224D831B7786C7E904F1EF2D9D004E0E825DD74AC4A576CDFCA7CECD14" +
            "D8E2E6CCAA3A302871AE0BA979BB25559215D771FAE647905878E797BBA9FC62" +
            "50F30F518A8008F5A12B35CE526E31032B56EFE5A4121E1E39DC7339A0CE8023" +
            "24CDDB7E9497BA37D8B9F8D826F901C52708935B4CA5B0D4D760A9FB33B0442D" +
            "008444D5AEB16E5C32187C7038F29160DD1A2D4DB1F9E9A6C035CF5BCED45287" +
            "C5DEBAB18743AAF90E77201FEA67485BA3BBCE90CEA4180C447EE588AC19C855" +
            "638B9552D47933D2760351174D9C3493DCCE9708B3EFE4BE398BA64051BF52B7" +
            "C1DCA44D2D0ED5A6CFB116DDA41995FA99373C254F3F3EBF0F0049F1159A8A76" +
            "4CFE9F9CC56C5489DD0F4E924158C9B1B626030CB492489F6AD0A9DCAF3E141D" +
            "B4D4821B2D8A384110B6B0B522F62A9DC0C1315A2A73A7F25F96C530E2F700F9" +
            "86829A839B944AE6758B8DD1A1E9257F91C160878A255E299C18424EB9983EDE" +
            "6DD1C5F4D5453DD5A56AC87DB1EFA0806E3DBFF10A9623FBAA0BAF352F50AB5D" +
            "B16AB1171145860D21E2AB20B45C8865B48390A66057DE3A1ABE45EA65376EF6" +
            "A96FE36285C2328C318182301306092A864886F70D0109153106040401000000" +
            "306B06092B0601040182371101315E1E5C004D006900630072006F0073006F00" +
            "66007400200045006E00680061006E0063006500640020004300720079007000" +
            "74006F0067007200610070006800690063002000500072006F00760069006400" +
            "650072002000760031002E003030820FAF06092A864886F70D010706A0820FA0" +
            "30820F9C02010030820F9506092A864886F70D010701301C060A2A864886F70D" +
            "010C0106300E0408FFCC41FD8C8414F6020207D080820F68092C6010873CF9EC" +
            "54D4676BCFB5FA5F523D03C981CB4A3DC096074E7D04365DDD1E80BF366B8F9E" +
            "C4BC056E8CE0CAB516B9C28D17B55E1EB744C43829D0E06217852FA99CCF5496" +
            "176DEF9A48967C1EEB4A384DB7783E643E35B5B9A50533B76B8D53581F02086B" +
            "782895097860D6CA512514E10D004165C85E561DF5F9AEFD2D89B64F178A7385" +
            "C7FA40ECCA899B4B09AE40EE60DAE65B31FF2D1EE204669EFF309A1C7C8D7B07" +
            "51AE57276D1D0FB3E8344A801AC5226EA4ED97FCD9399A4EB2E778918B81B17F" +
            "E4F65B502595195C79E6B0E37EB8BA36DB12435587E10037D31173285D45304F" +
            "6B0056512B3E147D7B5C397709A64E1D74F505D2BD72ED99055161BC57B6200F" +
            "2F48CF128229EFBEBFC2707678C0A8C51E3C373271CB4FD8EF34A1345696BF39" +
            "50E8CE9831F667D68184F67FE4D30332E24E5C429957694AF23620EA7742F08A" +
            "38C9A517A7491083A367B31C60748D697DFA29635548C605F898B64551A48311" +
            "CB2A05B1ACA8033128D48E4A5AA263D970FE59FBA49017F29049CF80FFDBD192" +
            "95B421FEFF6036B37D2F8DC8A6E36C4F5D707FB05274CC0D8D94AFCC8C6AF546" +
            "A0CF49FBD3A67FB6D20B9FE6FDA6321E8ABF5F7CC794CFCC46005DC57A7BAFA8" +
            "9954E43230402C8100789F11277D9F05C78DF0509ECFBF3A85114FD35F4F17E7" +
            "98D60C0008064E2557BA7BF0B6F8663A6C014E0220693AE29E2AB4BDE5418B61" +
            "0889EC02FF5480BD1B344C87D73E6E4DB98C73F881B22C7D298059FE9D7ADA21" +
            "92BB6C87F8D25F323A70D234E382F6C332FEF31BB11C37E41903B9A59ADEA5E0" +
            "CBAB06DFB835257ABC179A897DEAD9F19B7DF861BE94C655DC73F628E065F921" +
            "E5DE98FFCBDF2A54AC01E677E365DD8B932B5BDA761A0032CE2127AB2A2B9DCB" +
            "63F1EA8A51FC360AB5BC0AD435F21F9B6842980D795A6734FDB27A4FA8209F73" +
            "62DD632FC5FB1F6DE762473D6EA68BFC4BCF983865E66E6D93159EFACC40AB31" +
            "AA178806CF893A76CAAA3279C988824A33AF734FAF8E21020D988640FAB6DB10" +
            "DF21D93D01776EEA5DAECF695E0C690ED27AD386E6F2D9C9482EA38946008CCB" +
            "8F0BD08F9D5058CF8057CA3AD50BB537116A110F3B3ACD9360322DB4D242CC1A" +
            "6E15FA2A95192FC65886BE2672031D04A4FB0B1F43AE8476CF82638B61B416AA" +
            "97925A0110B736B4D83D7977456F35D947B3D6C9571D8E2DA0E9DEE1E665A844" +
            "259C17E01E044FAB898AA170F99157F7B525D524B01BD0710D23A7689A615703" +
            "8A0697BD48FFE0253ABD6F862093574B2FC9BA38E1A6EC60AF187F10D79FF71F" +
            "7C50E87A07CC0A51099899F7336FE742ADEF25E720B8E0F8781EC7957D414CF5" +
            "D44D6998E7E35D2433AFD86442CCA637A1513BE3020B5334614277B3101ED7AD" +
            "22AFE50DE99A2AD0E690596C93B881E2962D7E52EE0A770FAF6917106A8FF029" +
            "8DF38D6DE926C30834C5D96854FFD053BDB020F7827FB81AD04C8BC2C773B2A5" +
            "9FDD6DDF7298A052B3486E03FECA5AA909479DDC7FED972192792888F49C40F3" +
            "910140C5BE264D3D07BEBF3275117AF51A80C9F66C7028A2C3155414CF939997" +
            "268A1F0AA9059CC3AA7C8BBEF880187E3D1BA8978CBB046E43289A020CAE11B2" +
            "5140E2247C15A32CF70C7AA186CBB68B258CF2397D2971F1632F6EBC4846444D" +
            "E445673B942F1F110C7D586B6728ECA5B0A62D77696BF25E21ED9196226E5BDA" +
            "5A80ECCC785BEEDE917EBC6FFDC2F7124FE8F719B0A937E35E9A720BB9ED72D2" +
            "1213E68F058D80E9F8D7162625B35CEC4863BD47BC2D8D80E9B9048811BDD8CB" +
            "B70AB215962CD9C40D56AE50B7003630AE26341C6E243B3D12D5933F73F78F15" +
            "B014C5B1C36B6C9F410A77CA997931C8BD5CCB94C332F6723D53A4CCC630BFC9" +
            "DE96EFA7FDB66FA519F967D6A2DB1B4898BB188DEB98A41FFA7907AE7601DDE2" +
            "30E241779A0FDF551FB84D80AAEE3D979F0510CD026D4AE2ED2EFB7468418CCD" +
            "B3BD2A29CD7C7DC6419B4637412304D5DA2DC178C0B4669CA8330B9713A812E6" +
            "52E812135D807E361167F2A6814CEF2A8A9591EFE2C18216A517473B9C3BF2B7" +
            "51E47844893DA30F7DCD4222D1A55D570C1B6F6A99AD1F9213BA8F84C0B14A6D" +
            "ED6A26EAFF8F89DF733EEB44117DF0FD357186BA4A15BD5C669F60D6D4C34028" +
            "322D4DDF035302131AB6FD08683804CC90C1791182F1AE3281EE69DDBBCC12B8" +
            "1E60942FD082286B16BE27DC11E3BB0F18C281E02F3BA66E48C5FD8E8EA3B731" +
            "BDB12A4A3F2D9E1F833DD204372003532E1BB11298BDF5092F2959FC439E6BD2" +
            "DC6C37E3E775DCBE821B9CBB02E95D84C15E736CEA2FDDAD63F5CD47115B4AD5" +
            "5227C2A02886CD2700540EBFD5BF18DC5F94C5874972FD5424FE62B30500B1A8" +
            "7521EA3798D11970220B2BE7EFC915FCB7A6B8962F09ABA005861E839813EDA3" +
            "E59F70D1F9C277B73928DFFC84A1B7B0F78A8B001164EB0824F2510885CA269F" +
            "DCBB2C3AE91BDE91A8BBC648299A3EB626E6F4236CCE79E14C803498562BAD60" +
            "28F5B619125F80925A2D3B1A56790795D04F417003A8E9E53320B89D3A3109B1" +
            "9BB17B34CC9700DA138FABB5997EC34D0A44A26553153DBCFF8F6A1B5432B150" +
            "58F7AD87C6B37537796C95369DAD53BE5543D86D940892F93983153B4031D4FA" +
            "B25DAB02C1091ACC1DAE2118ABD26D19435CD4F1A02BDE1896236C174743BCA6" +
            "A33FB5429E627EB3FD9F513E81F7BD205B81AAE627C69CF227B043722FA05141" +
            "39347D202C9B7B4E55612FC27164F3B5F287F29C443793E22F6ED6D2F353ED82" +
            "A9F33EDBA8F5F1B2958F1D6A3943A9614E7411FDBCA597965CD08A8042307081" +
            "BAC5A070B467E52D5B91CA58F986C5A33502236B5BAE6DB613B1A408D16B29D3" +
            "560F1E94AD840CFA93E83412937A115ABF68322538DA8082F0192D19EAAA41C9" +
            "299729D487A9404ECDB6396DDA1534841EAE1E7884FA43574E213AE656116D9E" +
            "F7591AA7BDE2B44733DFE27AA59949E5DC0EE00FDF42130A748DDD0FB0053C1A" +
            "55986983C8B9CEAC023CAD7EDFFA1C20D3C437C0EF0FC9868D845484D8BE6538" +
            "EAADA6365D48BA776EE239ED045667B101E3798FE53E1D4B9A2ACBBE6AF1E5C8" +
            "8A3FB03AD616404013E249EC34458F3A7C9363E7772151119FE058BD0939BAB7" +
            "64A2E545B0B2FDAA650B7E849C8DD4033922B2CAE46D0461C04A2C87657CB4C0" +
            "FFBA23DED69D097109EC8BFDC25BB64417FEEB32842DE3EFEF2BF4A47F08B9FC" +
            "D1907BC899CA9DA604F5132FB420C8D142D132E7E7B5A4BD0EF4A56D9E9B0ACD" +
            "88F0E862D3F8F0440954879FFE3AA7AA90573C6BFDC6D6474C606ACA1CD94C1C" +
            "3404349DD83A639B786AFCDEA1779860C05400E0479708F4A9A0DD51429A3F35" +
            "FBD5FB9B68CECC1D585F3E35B7BBFC469F3EAEEB8020A6F0C8E4D1804A3EB32E" +
            "B3909E80B0A41571B23931E164E0E1D0D05379F9FD3BF51AF04D2BE78BDB84BD" +
            "787D419E85626297CB35FCFB6ED64042EAD2EBC17BB65677A1A33A5C48ADD280" +
            "237FB2451D0EFB3A3C32354222C7AB77A3C92F7A45B5FB10092698D88725864A" +
            "3685FBDD0DC741424FCCD8A00B928F3638150892CAAB535CC2813D13026615B9" +
            "9977F7B8240E914ACA0FF2DCB1A9274BA1F55DF0D24CCD2BAB7741C9EA8B1ECD" +
            "E97477C45F88F034FDF73023502944AEE1FF370260C576992826C4B2E5CE9924" +
            "84E3B85170FCCAC3413DC0FF6F093593219E637F699A98BD29E8EE4550C128CA" +
            "182680FDA3B10BC07625734EE8A8274B43B170FC3AEC9AA58CD92709D388E166" +
            "AB4ADFD5A4876DC47C17DE51FDD42A32AF672515B6A81E7ABECFE748912B321A" +
            "FD0CBF4880298DD79403900A4002B5B436230EB6E49192DF49FAE0F6B60EBA75" +
            "A54592587C141AD3B319129006367E9532861C2893E7A2D0D2832DF4377C3184" +
            "5CB02A1D020282C3D2B7F77221F71FEA7FF0A988FEF15C4B2F6637159EEC5752" +
            "D8A7F4AB971117666A977370E754A4EB0DC52D6E8901DC60FCD87B5B6EF9A91A" +
            "F8D9A4E11E2FFDAB55FC11AF6EEB5B36557FC8945A1E291B7FF8931BE4A57B8E" +
            "68F04B9D4A9A02FC61AE913F2E2DDBEE42C065F4D30F568834D5BB15FDAF691F" +
            "197EF6C25AE87D8E968C6D15351093AAC4813A8E7B191F77E6B19146F839A43E" +
            "2F40DE8BE28EB22C0272545BADF3BD396D383B8DA8388147100B347999DDC412" +
            "5AB0AA1159BC6776BD2BF51534C1B40522D41466F414BDE333226973BAD1E6D5" +
            "7639D30AD94BEA1F6A98C047F1CE1294F0067B771778D59E7C722C73C2FF100E" +
            "13603206A694BF0ED07303BE0655DC984CA29893FD0A088B122B67AABDC803E7" +
            "3E5729E868B1CA26F5D05C818D9832C70F5992E7D15E14F9775C6AD24907CF2F" +
            "211CF87167861F94DCF9E3D365CB600B336D93AD44B8B89CA24E59C1F7812C84" +
            "DBE3EE57A536ED0D4BF948F7662E5BCBBB388C72243CFCEB720852D5A4A52F01" +
            "8C2C087E4DB43410FE9ABA3A8EF737B6E8FFDB1AB9832EBF606ED5E4BD62A86B" +
            "BCAE115C67682EDEA93E7845D0D6962C146B411F7784545851D2F327BEC7E434" +
            "4D68F137CDA217A3F0FF3B752A34C3B5339C79CB8E1AC690C038E85D6FC13379" +
            "090198D3555394D7A2159A23BD5EEF06EB0BCC729BB29B5BE911D02DA78FDA56" +
            "F035E508C722139AD6F25A6C84BED0E98893370164B033A2B52BC40D9BF5163A" +
            "F9650AB55EABB23370492A7D3A87E17C11B4D07A7296273F33069C835FD208BA" +
            "8F989A3CF8659054E2CCCFB0C983531DC6590F27C4A1D2C3A780FE945F7E52BB" +
            "9FFD2E324640E3E348541A620CD62605BBDB284AF97C621A00D5D1D2C31D6BD6" +
            "1149137B8A0250BC426417A92445A52574E999FB9102C16671914A1542E92DDE" +
            "541B2A0457112AF936DA84707CADFEA43BFEDAE5F58859908640420948086E57" +
            "FFD1B867C241D40197CB0D4AD58BB69B3724772E0079406A1272858AAA620668" +
            "F696955102639F3E95CFFC637EAF8AB54F0B5B2131AB292438D06E15F3826352" +
            "DEDC653DA5A4AACE2BB97061A498F3B6789A2310471B32F91A6B7A9944DDBB70" +
            "31525B3AE387214DC85A1C7749E9168F41272680D0B3C331D61175F23B623EEC" +
            "40F984C35C831268036680DE0821E5DEE5BB250C6984775D49B7AF94057371DB" +
            "72F81D2B0295FC6A51BCD00A697649D4346FDD59AC0DFAF21BFCC942C23C6134" +
            "FFBA2ABABC141FF700B52C5B26496BF3F42665A5B71BAC7F0C19870BD9873890" +
            "239C578CDDD8E08A1B0A429312FB24F151A11E4D180359A7FA043E8155453F67" +
            "265CB2812B1C98C144E7675CFC86413B40E35445AE7710227D13DC0B5550C870" +
            "10B363C492DA316FB40D3928570BF71BF47638F1401549369B1255DB080E5DFA" +
            "18EA666B9ECBE5C9768C06B3FF125D0E94B98BB24B4FD44E770B78D7B336E021" +
            "4FD72E77C1D0BE9F313EDCD147957E3463C62E753C10BB98584C85871AAEA9D1" +
            "F397FE9F1A639ADE31D40EAB391B03B588B8B031BCAC6C837C61B06E4B745052" +
            "474D33531086519C39EDD6310F3079EB5AC83289A6EDCBA3DC97E36E837134F7" +
            "303B301F300706052B0E03021A0414725663844329F8BF6DECA5873DDD8C96AA" +
            "8CA5D40414DF1D90CD18B3FBC72226B3C66EC2CB1AB351D4D2020207D0").HexToByteArray();

        internal static readonly byte[] Pkcs7ChainDerBytes = (
            "30820E1606092A864886F70D010702A0820E0730820E030201013100300B0609" +
            "2A864886F70D010701A0820DEB3082050B30820474A003020102020A15EAA83A" +
            "000100009291300D06092A864886F70D010105050030818131133011060A0992" +
            "268993F22C6401191603636F6D31193017060A0992268993F22C64011916096D" +
            "6963726F736F667431143012060A0992268993F22C6401191604636F72703117" +
            "3015060A0992268993F22C64011916077265646D6F6E643120301E0603550403" +
            "13174D532050617373706F7274205465737420537562204341301E170D313330" +
            "3131303231333931325A170D3331313231333232323630375A308185310B3009" +
            "060355040613025553310B30090603550408130257413110300E060355040713" +
            "075265646D6F6E64310D300B060355040A130454455354310D300B060355040B" +
            "130454455354311330110603550403130A746573742E6C6F63616C3124302206" +
            "092A864886F70D010901161563726973706F70406D6963726F736F66742E636F" +
            "6D30819F300D06092A864886F70D010101050003818D0030818902818100B406" +
            "851089E9CF7CDB438DD77BEBD819197BEEFF579C35EF9C4652DF9E6330AA7E2E" +
            "24B181C59DA4AF10E97220C1DF99F66CE6E97247E9126A016AC647BD2EFD136C" +
            "31470C7BE01A20E381243BEEC8530B7F6466C50A051DCE37274ED7FF2AFFF4E5" +
            "8AABA61D5A448F4A8A9B3765D1D769F627ED2F2DE9EE67B1A7ECA3D288C90203" +
            "010001A38202823082027E300E0603551D0F0101FF0404030204F0301D060355" +
            "1D250416301406082B0601050507030106082B06010505070302301D0603551D" +
            "0E04160414FB3485708CBF6188F720EF948489405C8D0413A7301F0603551D23" +
            "0418301680146A6678620A4FF49CA8B75FD566348F3371E42B133081D0060355" +
            "1D1F0481C83081C53081C2A081BFA081BC865F687474703A2F2F707074657374" +
            "73756263612E7265646D6F6E642E636F72702E6D6963726F736F66742E636F6D" +
            "2F43657274456E726F6C6C2F4D5325323050617373706F727425323054657374" +
            "25323053756225323043412831292E63726C865966696C653A2F2F5C5C707074" +
            "65737473756263612E7265646D6F6E642E636F72702E6D6963726F736F66742E" +
            "636F6D5C43657274456E726F6C6C5C4D532050617373706F7274205465737420" +
            "5375622043412831292E63726C3082013806082B060105050701010482012A30" +
            "82012630819306082B06010505073002868186687474703A2F2F707074657374" +
            "73756263612E7265646D6F6E642E636F72702E6D6963726F736F66742E636F6D" +
            "2F43657274456E726F6C6C2F70707465737473756263612E7265646D6F6E642E" +
            "636F72702E6D6963726F736F66742E636F6D5F4D5325323050617373706F7274" +
            "2532305465737425323053756225323043412831292E63727430818D06082B06" +
            "01050507300286818066696C653A2F2F5C5C70707465737473756263612E7265" +
            "646D6F6E642E636F72702E6D6963726F736F66742E636F6D5C43657274456E72" +
            "6F6C6C5C70707465737473756263612E7265646D6F6E642E636F72702E6D6963" +
            "726F736F66742E636F6D5F4D532050617373706F727420546573742053756220" +
            "43412831292E637274300D06092A864886F70D0101050500038181009DEBB8B5" +
            "A41ED54859795F68EF767A98A61EF7B07AAC190FCC0275228E4CAD360C9BA98B" +
            "0AE153C75522EEF42D400E813B4E49E7ACEB963EEE7B61D3C8DA05C183471544" +
            "725B2EBD1889877F62134827FB5993B8FDF618BD421ABA18D70D1C5B41ECDD11" +
            "695A48CB42EB501F96DA905471830C612B609126559120F6E18EA44830820358" +
            "308202C1A00302010202101B9671A4BC128B8341B0E314EAD9A191300D06092A" +
            "864886F70D01010505003081A13124302206092A864886F70D01090116156173" +
            "6D656D6F6E406D6963726F736F66742E636F6D310B3009060355040613025553" +
            "310B30090603550408130257413110300E060355040713075265646D6F6E6431" +
            "123010060355040A13094D6963726F736F667431163014060355040B130D5061" +
            "7373706F727420546573743121301F060355040313184D532050617373706F72" +
            "74205465737420526F6F74204341301E170D3035303132363031333933325A17" +
            "0D3331313231333232323630375A3081A13124302206092A864886F70D010901" +
            "161561736D656D6F6E406D6963726F736F66742E636F6D310B30090603550406" +
            "13025553310B30090603550408130257413110300E060355040713075265646D" +
            "6F6E6431123010060355040A13094D6963726F736F667431163014060355040B" +
            "130D50617373706F727420546573743121301F060355040313184D5320506173" +
            "73706F7274205465737420526F6F7420434130819F300D06092A864886F70D01" +
            "0101050003818D0030818902818100C4673C1226254F6BBD01B01D21BB05264A" +
            "9AA5B77AC51748EAC52048706DA6B890DCE043C6426FC44E76D70F9FE3A4AC85" +
            "5F533E3D08E140853DB769EE24DBDB7269FABEC0FDFF6ADE0AA85F0085B78864" +
            "58E7585E433B0924E81600433CB1177CE6AD5F2477B2A0E2D1A34B41F6C6F5AD" +
            "E4A9DD7D565C65F02C2AAA01C8E0C10203010001A3818E30818B301306092B06" +
            "0104018237140204061E0400430041300B0603551D0F040403020186300F0603" +
            "551D130101FF040530030101FF301D0603551D0E04160414F509C1D6267FC39F" +
            "CA1DE648C969C74FB111FE10301206092B060104018237150104050203010002" +
            "302306092B0601040182371502041604147F7A5208411D4607C0057C98F0C473" +
            "07010CB3DE300D06092A864886F70D0101050500038181004A8EAC73D8EA6D7E" +
            "893D5880945E0E3ABFC79C40BFA60A680CF8A8BF63EDC3AD9C11C081F1F44408" +
            "9581F5C8DCB23C0AEFA27571D971DBEB2AA9A1B3F7B9B0877E9311D36098A65B" +
            "7D03FC69A835F6C3096DEE135A864065F9779C82DEB0C777B9C4DB49F0DD11A0" +
            "EAB287B6E352F7ECA467D0D3CA2A8081119388BAFCDD25573082057C308204E5" +
            "A003020102020A6187C7F200020000001B300D06092A864886F70D0101050500" +
            "3081A13124302206092A864886F70D010901161561736D656D6F6E406D696372" +
            "6F736F66742E636F6D310B3009060355040613025553310B3009060355040813" +
            "0257413110300E060355040713075265646D6F6E6431123010060355040A1309" +
            "4D6963726F736F667431163014060355040B130D50617373706F727420546573" +
            "743121301F060355040313184D532050617373706F7274205465737420526F6F" +
            "74204341301E170D3039313032373231333133395A170D333131323133323232" +
            "3630375A30818131133011060A0992268993F22C6401191603636F6D31193017" +
            "060A0992268993F22C64011916096D6963726F736F667431143012060A099226" +
            "8993F22C6401191604636F727031173015060A0992268993F22C640119160772" +
            "65646D6F6E643120301E060355040313174D532050617373706F727420546573" +
            "742053756220434130819F300D06092A864886F70D010101050003818D003081" +
            "8902818100A6A4918F93C5D23B3C3A325AD8EC77043D207A0DDC294AD3F5BDE0" +
            "4033FADD4097BB1DB042B1D3B2F26A42CC3CB88FA9357710147AB4E1020A0DFB" +
            "2597AB8031DB62ABDC48398067EB79E4E2BBE5762F6B4C5EA7629BAC23F70269" +
            "06D46EC106CC6FBB4D143F7D5ADADEDE19B021EEF4A6BCB9D01DAEBB9A947703" +
            "40B748A3490203010001A38202D7308202D3300F0603551D130101FF04053003" +
            "0101FF301D0603551D0E041604146A6678620A4FF49CA8B75FD566348F3371E4" +
            "2B13300B0603551D0F040403020186301206092B060104018237150104050203" +
            "010001302306092B060104018237150204160414A0A485AE8296EA4944C6F6F3" +
            "886A8603FD07472C301906092B0601040182371402040C1E0A00530075006200" +
            "430041301F0603551D23041830168014F509C1D6267FC39FCA1DE648C969C74F" +
            "B111FE103081D60603551D1F0481CE3081CB3081C8A081C5A081C28663687474" +
            "703A2F2F70617373706F72747465737463612E7265646D6F6E642E636F72702E" +
            "6D6963726F736F66742E636F6D2F43657274456E726F6C6C2F4D532532305061" +
            "7373706F727425323054657374253230526F6F7425323043412831292E63726C" +
            "865B66696C653A2F2F50415353504F52545445535443412E7265646D6F6E642E" +
            "636F72702E6D6963726F736F66742E636F6D2F43657274456E726F6C6C2F4D53" +
            "2050617373706F7274205465737420526F6F742043412831292E63726C308201" +
            "4406082B06010505070101048201363082013230819A06082B06010505073002" +
            "86818D687474703A2F2F70617373706F72747465737463612E7265646D6F6E64" +
            "2E636F72702E6D6963726F736F66742E636F6D2F43657274456E726F6C6C2F50" +
            "415353504F52545445535443412E7265646D6F6E642E636F72702E6D6963726F" +
            "736F66742E636F6D5F4D5325323050617373706F727425323054657374253230" +
            "526F6F7425323043412832292E63727430819206082B06010505073002868185" +
            "66696C653A2F2F50415353504F52545445535443412E7265646D6F6E642E636F" +
            "72702E6D6963726F736F66742E636F6D2F43657274456E726F6C6C2F50415353" +
            "504F52545445535443412E7265646D6F6E642E636F72702E6D6963726F736F66" +
            "742E636F6D5F4D532050617373706F7274205465737420526F6F742043412832" +
            "292E637274300D06092A864886F70D010105050003818100C44788F8C4F5C2DC" +
            "84976F66417CBAE19FBFA82C257DA4C7FED6267BC711D113C78B1C097154A62A" +
            "B462ADC84A434AEBAE38DEB9605FAB534A3CAF7B72C199448E58640388911296" +
            "115ED6B3478D0E741D990F2D59D66F12E58669D8983489AB0406E37462164B56" +
            "6AA1D9B273C406FA694A2556D1D3ACE723382C19871B8C143100").HexToByteArray();

        internal static readonly byte[] Pkcs7ChainPemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN PKCS7-----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-----END PKCS7-----");

        internal static readonly byte[] Pkcs7EmptyPemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN PKCS7-----
MCcGCSqGSIb3DQEHAqAaMBgCAQExADALBgkqhkiG9w0BBwGgAKEAMQA=
-----END PKCS7-----");

        internal static readonly byte[] Pkcs7EmptyDerBytes = (
            "302706092A864886F70D010702A01A30180201013100300B06092A864886F70D" +
            "010701A000A1003100").HexToByteArray();

        internal static readonly byte[] Pkcs7SingleDerBytes = (
            "3082021406092A864886F70D010702A0820205308202010201013100300B0609" +
            "2A864886F70D010701A08201E9308201E530820152A0030201020210D5B5BC1C" +
            "458A558845BFF51CB4DFF31C300906052B0E03021D05003011310F300D060355" +
            "040313064D794E616D65301E170D3130303430313038303030305A170D313130" +
            "3430313038303030305A3011310F300D060355040313064D794E616D6530819F" +
            "300D06092A864886F70D010101050003818D0030818902818100B11E30EA8742" +
            "4A371E30227E933CE6BE0E65FF1C189D0D888EC8FF13AA7B42B68056128322B2" +
            "1F2B6976609B62B6BC4CF2E55FF5AE64E9B68C78A3C2DACC916A1BC7322DD353" +
            "B32898675CFB5B298B176D978B1F12313E3D865BC53465A11CCA106870A4B5D5" +
            "0A2C410938240E92B64902BAEA23EB093D9599E9E372E48336730203010001A3" +
            "46304430420603551D01043B3039801024859EBF125E76AF3F0D7979B4AC7A96" +
            "A1133011310F300D060355040313064D794E616D658210D5B5BC1C458A558845" +
            "BFF51CB4DFF31C300906052B0E03021D0500038181009BF6E2CF830ED485B86D" +
            "6B9E8DFFDCD65EFC7EC145CB9348923710666791FCFA3AB59D689FFD7234B787" +
            "2611C5C23E5E0714531ABADB5DE492D2C736E1C929E648A65CC9EB63CD84E57B" +
            "5909DD5DDF5DBBBA4A6498B9CA225B6E368B94913BFC24DE6B2BD9A26B192B95" +
            "7304B89531E902FFC91B54B237BB228BE8AFCDA264763100").HexToByteArray();

        internal static readonly byte[] Pkcs7SinglePemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN PKCS7-----
MIICFAYJKoZIhvcNAQcCoIICBTCCAgECAQExADALBgkqhkiG9w0BBwGgggHpMIIB
5TCCAVKgAwIBAgIQ1bW8HEWKVYhFv/UctN/zHDAJBgUrDgMCHQUAMBExDzANBgNV
BAMTBk15TmFtZTAeFw0xMDA0MDEwODAwMDBaFw0xMTA0MDEwODAwMDBaMBExDzAN
BgNVBAMTBk15TmFtZTCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEAsR4w6odC
SjceMCJ+kzzmvg5l/xwYnQ2Ijsj/E6p7QraAVhKDIrIfK2l2YJtitrxM8uVf9a5k
6baMeKPC2syRahvHMi3TU7MomGdc+1spixdtl4sfEjE+PYZbxTRloRzKEGhwpLXV
CixBCTgkDpK2SQK66iPrCT2VmenjcuSDNnMCAwEAAaNGMEQwQgYDVR0BBDswOYAQ
JIWevxJedq8/DXl5tKx6lqETMBExDzANBgNVBAMTBk15TmFtZYIQ1bW8HEWKVYhF
v/UctN/zHDAJBgUrDgMCHQUAA4GBAJv24s+DDtSFuG1rno3/3NZe/H7BRcuTSJI3
EGZnkfz6OrWdaJ/9cjS3hyYRxcI+XgcUUxq6213kktLHNuHJKeZIplzJ62PNhOV7
WQndXd9du7pKZJi5yiJbbjaLlJE7/CTeayvZomsZK5VzBLiVMekC/8kbVLI3uyKL
6K/NomR2MQA=
-----END PKCS7-----");

        internal static readonly byte[] MicrosoftDotComSslCertBytes = (
            "308205943082047CA00302010202103DF70C5D9903F8D8868B9B8CCF20DF6930" +
            "0D06092A864886F70D01010B05003077310B3009060355040613025553311D30" +
            "1B060355040A131453796D616E74656320436F72706F726174696F6E311F301D" +
            "060355040B131653796D616E746563205472757374204E6574776F726B312830" +
            "260603550403131F53796D616E74656320436C61737320332045562053534C20" +
            "4341202D204733301E170D3134313031353030303030305A170D313631303135" +
            "3233353935395A3082010F31133011060B2B0601040182373C02010313025553" +
            "311B3019060B2B0601040182373C0201020C0A57617368696E67746F6E311D30" +
            "1B060355040F131450726976617465204F7267616E697A6174696F6E31123010" +
            "06035504051309363030343133343835310B3009060355040613025553310E30" +
            "0C06035504110C0539383035323113301106035504080C0A57617368696E6774" +
            "6F6E3110300E06035504070C075265646D6F6E643118301606035504090C0F31" +
            "204D6963726F736F667420576179311E301C060355040A0C154D6963726F736F" +
            "667420436F72706F726174696F6E310E300C060355040B0C054D53434F4D311A" +
            "301806035504030C117777772E6D6963726F736F66742E636F6D30820122300D" +
            "06092A864886F70D01010105000382010F003082010A0282010100A46861FA9D" +
            "5DB763633BF5A64EF6E7C2C2367F48D2D46643A22DFCFCCB24E58A14D0F06BDC" +
            "956437F2A56BA4BEF70BA361BF12964A0D665AFD84B0F7494C8FA4ABC5FCA2E0" +
            "17C06178AEF2CDAD1B5F18E997A14B965C074E8F564970607276B00583932240" +
            "FE6E2DD013026F9AE13D7C91CC07C4E1E8E87737DC06EF2B575B89D62EFE4685" +
            "9F8255A123692A706C68122D4DAFE11CB205A7B3DE06E553F7B95F978EF8601A" +
            "8DF819BF32040BDF92A0DE0DF269B4514282E17AC69934E8440A48AB9D1F5DF8" +
            "9A502CEF6DFDBE790045BD45E0C94E5CA8ADD76A013E9C978440FC8A9E2A9A49" +
            "40B2460819C3E302AA9C9F355AD754C86D3ED77DDAA3DA13810B4D0203010001" +
            "A38201803082017C30310603551D11042A302882117777772E6D6963726F736F" +
            "66742E636F6D821377777771612E6D6963726F736F66742E636F6D3009060355" +
            "1D1304023000300E0603551D0F0101FF0404030205A0301D0603551D25041630" +
            "1406082B0601050507030106082B0601050507030230660603551D20045F305D" +
            "305B060B6086480186F84501071706304C302306082B06010505070201161768" +
            "747470733A2F2F642E73796D63622E636F6D2F637073302506082B0601050507" +
            "020230191A1768747470733A2F2F642E73796D63622E636F6D2F727061301F06" +
            "03551D230418301680140159ABE7DD3A0B59A66463D6CF200757D591E76A302B" +
            "0603551D1F042430223020A01EA01C861A687474703A2F2F73722E73796D6362" +
            "2E636F6D2F73722E63726C305706082B06010505070101044B3049301F06082B" +
            "060105050730018613687474703A2F2F73722E73796D63642E636F6D30260608" +
            "2B06010505073002861A687474703A2F2F73722E73796D63622E636F6D2F7372" +
            "2E637274300D06092A864886F70D01010B0500038201010015F8505B627ED7F9" +
            "F96707097E93A51E7A7E05A3D420A5C258EC7A1CFE1843EC20ACF728AAFA7A1A" +
            "1BC222A7CDBF4AF90AA26DEEB3909C0B3FB5C78070DAE3D645BFCF840A4A3FDD" +
            "988C7B3308BFE4EB3FD66C45641E96CA3352DBE2AEB4488A64A9C5FB96932BA7" +
            "0059CE92BD278B41299FD213471BD8165F924285AE3ECD666C703885DCA65D24" +
            "DA66D3AFAE39968521995A4C398C7DF38DFA82A20372F13D4A56ADB21B582254" +
            "9918015647B5F8AC131CC5EB24534D172BC60218A88B65BCF71C7F388CE3E0EF" +
            "697B4203720483BB5794455B597D80D48CD3A1D73CBBC609C058767D1FF060A6" +
            "09D7E3D4317079AF0CD0A8A49251AB129157F9894A036487").HexToByteArray();

        internal static readonly byte[] MicrosoftDotComIssuerBytes = (
            "3082052B30820413A00302010202107EE14A6F6FEFF2D37F3FAD654D3ADAB430" +
            "0D06092A864886F70D01010B05003081CA310B30090603550406130255533117" +
            "3015060355040A130E566572695369676E2C20496E632E311F301D060355040B" +
            "1316566572695369676E205472757374204E6574776F726B313A303806035504" +
            "0B1331286329203230303620566572695369676E2C20496E632E202D20466F72" +
            "20617574686F72697A656420757365206F6E6C79314530430603550403133C56" +
            "6572695369676E20436C6173732033205075626C6963205072696D6172792043" +
            "657274696669636174696F6E20417574686F72697479202D204735301E170D31" +
            "33313033313030303030305A170D3233313033303233353935395A3077310B30" +
            "09060355040613025553311D301B060355040A131453796D616E74656320436F" +
            "72706F726174696F6E311F301D060355040B131653796D616E74656320547275" +
            "7374204E6574776F726B312830260603550403131F53796D616E74656320436C" +
            "61737320332045562053534C204341202D20473330820122300D06092A864886" +
            "F70D01010105000382010F003082010A0282010100D8A1657423E82B64E232D7" +
            "33373D8EF5341648DD4F7F871CF84423138EFB11D8445A18718E601626929BFD" +
            "170BE1717042FEBFFA1CC0AAA3A7B571E8FF1883F6DF100A1362C83D9CA7DE2E" +
            "3F0CD91DE72EFB2ACEC89A7F87BFD84C041532C9D1CC9571A04E284F84D935FB" +
            "E3866F9453E6728A63672EBE69F6F76E8E9C6004EB29FAC44742D27898E3EC0B" +
            "A592DCB79ABD80642B387C38095B66F62D957A86B2342E859E900E5FB75DA451" +
            "72467013BF67F2B6A74D141E6CB953EE231A4E8D48554341B189756A4028C57D" +
            "DDD26ED202192F7B24944BEBF11AA99BE3239AEAFA33AB0A2CB7F46008DD9F1C" +
            "CDDD2D016680AFB32F291D23B88AE1A170070C340F0203010001A382015D3082" +
            "0159302F06082B0601050507010104233021301F06082B060105050730018613" +
            "687474703A2F2F73322E73796D63622E636F6D30120603551D130101FF040830" +
            "060101FF02010030650603551D20045E305C305A0604551D2000305230260608" +
            "2B06010505070201161A687474703A2F2F7777772E73796D617574682E636F6D" +
            "2F637073302806082B06010505070202301C1A1A687474703A2F2F7777772E73" +
            "796D617574682E636F6D2F72706130300603551D1F042930273025A023A02186" +
            "1F687474703A2F2F73312E73796D63622E636F6D2F706361332D67352E63726C" +
            "300E0603551D0F0101FF04040302010630290603551D1104223020A41E301C31" +
            "1A30180603550403131153796D616E746563504B492D312D353333301D060355" +
            "1D0E041604140159ABE7DD3A0B59A66463D6CF200757D591E76A301F0603551D" +
            "230418301680147FD365A7C2DDECBBF03009F34339FA02AF333133300D06092A" +
            "864886F70D01010B050003820101004201557BD0161A5D58E8BB9BA84DD7F3D7" +
            "EB139486D67F210B47BC579B925D4F059F38A4107CCF83BE0643468D08BC6AD7" +
            "10A6FAABAF2F61A863F265DF7F4C8812884FB369D9FF27C00A97918F56FB89C4" +
            "A8BB922D1B73B0C6AB36F4966C2008EF0A1E6624454F670040C8075474333BA6" +
            "ADBB239F66EDA2447034FB0EEA01FDCF7874DFA7AD55B75F4DF6D63FE086CE24" +
            "C742A9131444354BB6DFC960AC0C7FD993214BEE9CE4490298D3607B5CBCD530" +
            "2F07CE4442C40B99FEE69FFCB07886516DD12C9DC696FB8582BB042FF76280EF" +
            "62DA7FF60EAC90B856BD793FF2806EA3D9B90F5D3A071D9193864B294CE1DCB5" +
            "E1E0339DB3CB36914BFEA1B4EEF0F9").HexToByteArray();

        internal static readonly byte[] MicrosoftDotComRootBytes = (
            "308204D3308203BBA003020102021018DAD19E267DE8BB4A2158CDCC6B3B4A30" +
            "0D06092A864886F70D01010505003081CA310B30090603550406130255533117" +
            "3015060355040A130E566572695369676E2C20496E632E311F301D060355040B" +
            "1316566572695369676E205472757374204E6574776F726B313A303806035504" +
            "0B1331286329203230303620566572695369676E2C20496E632E202D20466F72" +
            "20617574686F72697A656420757365206F6E6C79314530430603550403133C56" +
            "6572695369676E20436C6173732033205075626C6963205072696D6172792043" +
            "657274696669636174696F6E20417574686F72697479202D204735301E170D30" +
            "36313130383030303030305A170D3336303731363233353935395A3081CA310B" +
            "300906035504061302555331173015060355040A130E566572695369676E2C20" +
            "496E632E311F301D060355040B1316566572695369676E205472757374204E65" +
            "74776F726B313A3038060355040B133128632920323030362056657269536967" +
            "6E2C20496E632E202D20466F7220617574686F72697A656420757365206F6E6C" +
            "79314530430603550403133C566572695369676E20436C617373203320507562" +
            "6C6963205072696D6172792043657274696669636174696F6E20417574686F72" +
            "697479202D20473530820122300D06092A864886F70D01010105000382010F00" +
            "3082010A0282010100AF240808297A359E600CAAE74B3B4EDC7CBC3C451CBB2B" +
            "E0FE2902F95708A364851527F5F1ADC831895D22E82AAAA642B38FF8B955B7B1" +
            "B74BB3FE8F7E0757ECEF43DB66621561CF600DA4D8DEF8E0C362083D5413EB49" +
            "CA59548526E52B8F1B9FEBF5A191C23349D843636A524BD28FE870514DD18969" +
            "7BC770F6B3DC1274DB7B5D4B56D396BF1577A1B0F4A225F2AF1C926718E5F406" +
            "04EF90B9E400E4DD3AB519FF02BAF43CEEE08BEB378BECF4D7ACF2F6F03DAFDD" +
            "759133191D1C40CB7424192193D914FEAC2A52C78FD50449E48D6347883C6983" +
            "CBFE47BD2B7E4FC595AE0E9DD4D143C06773E314087EE53F9F73B8330ACF5D3F" +
            "3487968AEE53E825150203010001A381B23081AF300F0603551D130101FF0405" +
            "30030101FF300E0603551D0F0101FF040403020106306D06082B060105050701" +
            "0C0461305FA15DA05B3059305730551609696D6167652F6769663021301F3007" +
            "06052B0E03021A04148FE5D31A86AC8D8E6BC3CF806AD448182C7B192E302516" +
            "23687474703A2F2F6C6F676F2E766572697369676E2E636F6D2F76736C6F676F" +
            "2E676966301D0603551D0E041604147FD365A7C2DDECBBF03009F34339FA02AF" +
            "333133300D06092A864886F70D0101050500038201010093244A305F62CFD81A" +
            "982F3DEADC992DBD77F6A5792238ECC4A7A07812AD620E457064C5E797662D98" +
            "097E5FAFD6CC2865F201AA081A47DEF9F97C925A0869200DD93E6D6E3C0D6ED8" +
            "E606914018B9F8C1EDDFDB41AAE09620C9CD64153881C994EEA284290B136F8E" +
            "DB0CDD2502DBA48B1944D2417A05694A584F60CA7E826A0B02AA251739B5DB7F" +
            "E784652A958ABD86DE5E8116832D10CCDEFDA8822A6D281F0D0BC4E5E71A2619" +
            "E1F4116F10B595FCE7420532DBCE9D515E28B69E85D35BEFA57D4540728EB70E" +
            "6B0E06FB33354871B89D278BC4655F0D86769C447AF6955CF65D320833A454B6" +
            "183F685CF2424A853854835FD1E82CF2AC11D6A8ED636A").HexToByteArray();

        internal static readonly byte[] Rsa384CertificatePemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN CERTIFICATE-----
MIICTzCCAgmgAwIBAgIJAMQtYhFJ0+5jMA0GCSqGSIb3DQEBBQUAMIGSMQswCQYD
VQQGEwJVUzETMBEGA1UECAwKV2FzaGluZ3RvbjEQMA4GA1UEBwwHUmVkbW9uZDEY
MBYGA1UECgwPTWljcm9zb2Z0IENvcnAuMSAwHgYDVQQLDBcuTkVUIEZyYW1ld29y
ayAoQ29yZUZ4KTEgMB4GA1UEAwwXUlNBIDM4NC1iaXQgQ2VydGlmaWNhdGUwHhcN
MTYwMzAyMTY1OTA0WhcNMTYwNDAxMTY1OTA0WjCBkjELMAkGA1UEBhMCVVMxEzAR
BgNVBAgMCldhc2hpbmd0b24xEDAOBgNVBAcMB1JlZG1vbmQxGDAWBgNVBAoMD01p
Y3Jvc29mdCBDb3JwLjEgMB4GA1UECwwXLk5FVCBGcmFtZXdvcmsgKENvcmVGeCkx
IDAeBgNVBAMMF1JTQSAzODQtYml0IENlcnRpZmljYXRlMEwwDQYJKoZIhvcNAQEB
BQADOwAwOAIxANrMIthuZxV1Ay4x8gbc/BksZeLVEInlES0JbyiCr9tbeM22Vy/S
9h2zkEciMuPZ9QIDAQABo1AwTjAdBgNVHQ4EFgQU5FG2Fmi86hJOCf4KnjaxOGWV
dRUwHwYDVR0jBBgwFoAU5FG2Fmi86hJOCf4KnjaxOGWVdRUwDAYDVR0TBAUwAwEB
/zANBgkqhkiG9w0BAQUFAAMxAEzDg/u8TlApCnE8qxhcbTXk2MbX+2n5PCn+MVrW
wggvPj3b2WMXsVWiPr4S1Y/nBA==
-----END CERTIFICATE-----");

        internal static readonly byte[] ECDsabrainpoolP160r1_Pfx = (
            "308203D00201033082039606092A864886F70D010701A0820387048203833082" +
            "037F3082029F06092A864886F70D010706A08202903082028C02010030820285" +
            "06092A864886F70D010701301C060A2A864886F70D010C0106300E0408E4D7F5" +
            "F71CA4D5380202080080820258F19503250C93322C81CCC92C57AD76AD4DDF79" +
            "10DBAB6A63D6AAF3F470AC5283BDEB532086D3379B7A3D68D17FAC483EBEA029" +
            "1D2B5F862885E048A3034580A41F238AA836B9E94C12B0656B51C72355AED1DD" +
            "19FE771E3768095D6467FC8742BE0BC5D65360CD875D35C23D272842F64791A1" +
            "53F96AFBD3D7660EC016BF9D59B2B68C2A34D93B133697D6B77DB27649BEEABC" +
            "0B68D35DB3779DD4C871C9C26799E6ABB5E9048DDC44C6E6310F3A023AD09E97" +
            "1AB1DF38FDF3091FB35125EA3A14F5D72A00EC4C637951F026DE79C0E30E0244" +
            "808FB46EFD4EA9C67411DC2B13842B273F405F6D58D45D1D2D47BC1362ED9884" +
            "C2C44EA334A3B02C7E661F489DED798B63D64F90916596BADC87C68C868FCECB" +
            "6F246410186BBB2F2DC2ED24BF215AA54570072E21970CF856398BB1FD8C2F61" +
            "0788231C51D45CE471A235464147A405799F8CBE39AA30A8BFD2534C738AE330" +
            "8771394D871429EF2D6AB6381F793D7CBC0374D4E529B8B6AA37BE04FBE71A9A" +
            "A7954814C0C8841740539ED97DB56C3CBE5851C9D875E4B6023AE095B62C9DC5" +
            "36E06DA40C4B874776CBABDDAB50BDD5ECF9D997EEB1483D3AC23E6F37DD4CBD" +
            "64163E7A115BCE44554C53DD860B83CBE3B35F1E26B87185C602E4FFB3A99458" +
            "0D6A9334F74AA29B3609926FE86F197C955CBAEC2A41EE1572A4295F62D4D9CA" +
            "050CD933BC0BA43D7744EAFA9D6B8253241EB36C605DC334A6470BC709F13985" +
            "8AC238FD7F3C14EDDAB6E29996FE966A96EAC23CF17063C89315734D76CCB21F" +
            "94A7E4A44A5F6DCB93CEEEFB539664296F8F609CFE293200FE4B5EE57AB3A1E7" +
            "A3483DC6243081D906092A864886F70D010701A081CB0481C83081C53081C206" +
            "0B2A864886F70D010C0A0102A0818B308188301C060A2A864886F70D010C0103" +
            "300E0408CD30F3C5A9918832020208000468BF29F33058622BA159CDD3DE2B49" +
            "CBDD736BF1483FF4D43BACCC93B544A513D5E47DB4FECADBB4E3277A6B90345D" +
            "7E73F507924A615D968F807834D3796EFB0A3EF214A75883E3AB90086DA2418B" +
            "0B2D657DEA39A8600172B6975FFB39E88863DB11283A5CEA1FCB312530230609" +
            "2A864886F70D01091531160414472193B362B056F6D6928EFF4C43FF1EFEB173" +
            "4E30313021300906052B0E03021A05000414B703685D5039D8EEF1A46F772F31" +
            "F177FDE874EC0408B4EF89F18902CE9502020800").HexToByteArray();

        internal static readonly byte[] ECDsabrainpoolP160r1_Explicit_Pfx = (
            "30820501020103308204C706092A864886F70D010701A08204B8048204B43082" +
            "04B03082032F06092A864886F70D010706A08203203082031C02010030820315" +
            "06092A864886F70D010701301C060A2A864886F70D010C0106300E0408C55E7D" +
            "72B26355D202020800808202E8B8BCB9180C8751F860C5F77C04CC921A89D836" +
            "E8BC06DA99238EF8D65FB3473AF97D930B93471B28F08F483062BCEDB44287FA" +
            "E813B6CA812393475F50BB2D0AD233A8CE0832DECE54EF78A1F289A7802A9900" +
            "BAA2B7392BCF6C963D54D8DD63898D5D2FA472E77194763822157664417D93D8" +
            "BF740A4A5683FFFDF8B4CC7E074C6DCFF73E21368D743F0A1CE33484B02A4502" +
            "F7340A853F82F606B10DEA7239EF56C8DBDAED8DD3C956DD4D3E42CA517318E6" +
            "0DF70721469C512E120C500956D960A59EEB4A3A6541891319ACA68AB99462EA" +
            "CB59B52F6C6BFCF24EEA4E19BDDC5DDE1B6F36549DD589E1B10CAA02D4A53C36" +
            "1F1B9F101F9375394CF9BC8D941F94CC4F128EC861AA6B4182E1F81D2C0BADEA" +
            "2E6596BAC8819DE293ECEEC7C54BE97CD3917E0497F2559D44EF3484618DCB9B" +
            "85820762F392925BB36BD482DF544B614CDF9C6728BD92E858C1DF61435E7A03" +
            "DED27DA460603A61BE7DB93870E9530EB51384663E42A49C342891B8E757ED49" +
            "18A590D59734EA1C95E992CD93B80EBD3C246999305C204A813E0DCF26E64446" +
            "15A79E74042C7EAD4FEF0E68AA9DF627B5F4C037BF728015B6BBFA046CAA805C" +
            "BE7B1867262B36172E8759FAE7B965FF9B98D3539A48525E10A57A84C55AEFAC" +
            "3ED9025AB8B0680E0896DDD776C0AFC1A95BDD5DBE0ECCEB860B3CD3D6A2A493" +
            "2BC7774246A6936AFABA9BA8292841F9D6417AFFB00872E9B4ADF11889AEF20A" +
            "FCB8EAEBADAF38A2A240D36940B1585B37F7CA6A887EE1FBA199990FC104CD1F" +
            "A983CC2CE91156559EFCFC0F25B7C24B161DF4B4436F14428C4AE06F49FCC328" +
            "D00891A44AFAE5412FD330E23CFAE6107B4C965DFDB6335E8EFDF2C40767846B" +
            "C95ABF784DE155EED92DAB7A265DC17BC3ADA68D04E378D2E4E8924B05937092" +
            "E779EB04899E4FB19AAE7AA5FCF8D7A33BA464E4BB1FFB4E4D4CD71152F42B68" +
            "F5AB298B10DEB653C7F77F66B761DFD61E4E2DDD13B0B15639473BF5C3B8A31D" +
            "3D2334287F61E1A06F44CD3F2E74F59F43876F0D923082017906092A864886F7" +
            "0D010701A082016A04820166308201623082015E060B2A864886F70D010C0A01" +
            "02A082012630820122301C060A2A864886F70D010C0103300E0408A92CDE5D9A" +
            "F5278D0202080004820100A302E03B1BDF59D4ECD6F0FE7745F7C0DCE2CCEF0E" +
            "B26A7B92839B60288B05445BA1C91109D7E632E0C7B742E2D7D0947573AFEF1F" +
            "FAFCF8135DA3B5EE26A8E3AB7F415A8A19112724F850F024D3527F1FE2A303B1" +
            "34A644307AC6816E59E08540FA782351B27E37775AF3CD904E50A1A76C7C4F34" +
            "7EE78A1ED51FF71D00954130369012750746A280983E883E59AFDBBCCC7D1AA0" +
            "ECDCF2079ECFA4645E156ACC5FD6913763FC93C2E0C572042D00FE4EEB5E75DE" +
            "28C21FA1A7356C4071572DF23CC23833EA26705C0AA080636E27512B5F5755FE" +
            "A0514A31833D52C48A743BCDC0B58257FEDD23EE4EDBC06B574019E792B44BD6" +
            "3F3770875A25075183AF2C3125302306092A864886F70D01091531160414141C" +
            "1C8591A700DDE70FAC750C1539B2DFECAA3C30313021300906052B0E03021A05" +
            "0004143706E218219A899C700BD7AE3E8650FD1B2885AB0408E77FDD798BCADE" +
            "3C02020800").HexToByteArray();

        internal static readonly byte[] ECDsabrainpoolP160r1_CertificatePemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN CERTIFICATE-----
MIIB+TCCAbagAwIBAgIJAIJ8gBL6U36GMAoGCCqGSM49BAMCMHAxCzAJBgNVBAYT
AlVTMRUwEwYDVQQIDAxOb3J0aCBEYWtvdGExDjAMBgNVBAcMBUZhcmdvMRgwFgYD
VQQKDA9NaWNyb3NvZnQgQ29ycC4xIDAeBgNVBAsMFy5ORVQgRnJhbWV3b3JrIChD
b3JlRngpMB4XDTE2MDYxMDE0NTYzOFoXDTE2MDcxMDE0NTYzOFowcDELMAkGA1UE
BhMCVVMxFTATBgNVBAgMDE5vcnRoIERha290YTEOMAwGA1UEBwwFRmFyZ28xGDAW
BgNVBAoMD01pY3Jvc29mdCBDb3JwLjEgMB4GA1UECwwXLk5FVCBGcmFtZXdvcmsg
KENvcmVGeCkwQjAUBgcqhkjOPQIBBgkrJAMDAggBAQEDKgAEQk2dep8HoNJcbCal
ie5QIMYsNnphtOo9WUCgrrzEG3wfrxz39HcAXaNQME4wHQYDVR0OBBYEFPprBFD9
qDQynQJmJUpVKv9WR8z5MB8GA1UdIwQYMBaAFPprBFD9qDQynQJmJUpVKv9WR8z5
MAwGA1UdEwQFMAMBAf8wCgYIKoZIzj0EAwIDMQAwLgIVAN3U12PFcEe4HHi+Rio+
xk3lf6EbAhUAqdeGDOXgpHEoWEmzOQ6nWWQik1k=
-----END CERTIFICATE-----");

        internal static readonly byte[] ECDsabrainpoolP160r1_ExplicitCertificatePemBytes = ByteUtils.AsciiBytes(
            @"-----BEGIN CERTIFICATE-----
MIICijCCAkigAwIBAgIJAJVtMTsUqcjsMAoGCCqGSM49BAMCMHAxCzAJBgNVBAYT
AlVTMRUwEwYDVQQIDAxOb3J0aCBEYWtvdGExDjAMBgNVBAcMBUZhcmdvMRgwFgYD
VQQKDA9NaWNyb3NvZnQgQ29ycC4xIDAeBgNVBAsMFy5ORVQgRnJhbWV3b3JrIChD
b3JlRngpMB4XDTE2MDYxMDE1MDg1NVoXDTE2MDcxMDE1MDg1NVowcDELMAkGA1UE
BhMCVVMxFTATBgNVBAgMDE5vcnRoIERha290YTEOMAwGA1UEBwwFRmFyZ28xGDAW
BgNVBAoMD01pY3Jvc29mdCBDb3JwLjEgMB4GA1UECwwXLk5FVCBGcmFtZXdvcmsg
KENvcmVGeCkwgdMwgaQGByqGSM49AgEwgZgCAQEwIAYHKoZIzj0BAQIVAOleSl9z
cFncYN/HrZWz2BOVFWIPMCwEFDQOe+KigOt04r5hutp0XZfo98MABBQeWJqFlUI0
EhNPqi297JXI2GdeWAQpBL7VrxbqP2pPYpOMRjHrWve9vNvDFmfLR3oajsM4+UdB
ZpyXYxbaYyECFQDpXkpfc3BZ3GDfWZHUUClAnmD8CQIBAQMqAARz9ueEHonciPIW
lTsd673ZaNpP9GMoFfHns3DnUC0pC1Grh+6sZcPIo1AwTjAdBgNVHQ4EFgQU65OI
c9u4x/ZfIRyjcSaZTUKSsuIwHwYDVR0jBBgwFoAU65OIc9u4x/ZfIRyjcSaZTUKS
suIwDAYDVR0TBAUwAwEB/zAKBggqhkjOPQQDAgMwADAtAhUAxMT7z8lLv7hgWmGh
5siYmHkAExoCFFaaS2r7/kdkXsauyr37q6ewD6s+
-----END CERTIFICATE-----");

        internal static readonly ECDsaCngKeyValues ECDsaCng256PublicKey = 
            new ECDsaCngKeyValues()
            {
                QX = "448d98ee08aeba0d8b40f3c6dbd500e8b69f07c70c661771655228ea5a178a91".HexToByteArray(),
                QY = "0ef5cb1759f6f2e062021d4f973f5bb62031be87ae915cff121586809e3219af".HexToByteArray(),
                D = "692837e9cf613c0e290462a6f08faadcc7002398f75598d5554698a0cb51cf47".HexToByteArray(),
            };
    
        internal static readonly byte[] ECDsa256Certificate =
             ("308201223081c9a00302010202106a3c9e85ba6af1ac4f08111d8bdda340300906072a8648ce3d0401301431123010060355"
            + "04031309456332353655736572301e170d3135303931303231333533305a170d3136303931303033333533305a3014311230"
            + "10060355040313094563323536557365723059301306072a8648ce3d020106082a8648ce3d03010703420004448d98ee08ae"
            + "ba0d8b40f3c6dbd500e8b69f07c70c661771655228ea5a178a910ef5cb1759f6f2e062021d4f973f5bb62031be87ae915cff"
            + "121586809e3219af300906072a8648ce3d04010349003046022100f221063dca71955d17c8f0e0f63a144c4065578fd9f68e"
            + "1ae6a7683e209ea742022100ed1db6a8be27cfb20ab43e0ca061622ceff26f7249a0f791e4d6be1a4e52adfa").HexToByteArray();

        internal static readonly ECDsaCngKeyValues ECDsaCng384PublicKey =
            new ECDsaCngKeyValues()
            {
                QX = "c59eca607aa5559e6b2f8ac2eeb12d9ab47f420feabeb444c3f71520d7f2280439979323ab5a67344811d296fef6d1bd".HexToByteArray(),
                QY = "d15f307cc6cc6c8baeeeb168bfb02c34d6eb0621efb3d06ad31c06b29eaf6ec2ec67bf288455e729d82e5a6439f70901".HexToByteArray(),
                D = "f55ba33e28cea32a014e2fe1213bb4d41cef361f1fee022116b15be50feb96bc946b10a46a9a7a94176787e0928a3e1d".HexToByteArray(),
            };

        internal static readonly byte[] ECDsa384Certificate =
             ("3082015f3081e6a00302010202101e78eb573e70a2a64744672296988ad7300906072a8648ce3d0401301431123010060355"
            + "04031309456333383455736572301e170d3135303931303231333634365a170d3136303931303033333634365a3014311230"
            + "10060355040313094563333834557365723076301006072a8648ce3d020106052b8104002203620004c59eca607aa5559e6b"
            + "2f8ac2eeb12d9ab47f420feabeb444c3f71520d7f2280439979323ab5a67344811d296fef6d1bdd15f307cc6cc6c8baeeeb1"
            + "68bfb02c34d6eb0621efb3d06ad31c06b29eaf6ec2ec67bf288455e729d82e5a6439f70901300906072a8648ce3d04010369"
            + "003066023100a8fbaeeae61953897eae5f0beeeffaca48e89bc0cb782145f39f4ba5b03390ce6a28e432e664adf5ebc6a802"
            + "040b238b023100dcc19109383b9482fdda68f40a63ee41797dbb8f25c0284155cc4238d682fbb3fb6e86ea0933297e850a26"
            + "16f6c39bbf").HexToByteArray();

        internal static readonly ECDsaCngKeyValues ECDsaCng521PublicKey =
            new ECDsaCngKeyValues()
            {
                QX = "0134af29d1fe5e581fd2ff6194263abcb6f8cb4d9c08bdb384ede9b8663ae2f4e1af6c85eacc69dc768fbfcd856630792e05484cefb1fefb693081dc6490dac579c0".HexToByteArray(),
                QY = "00bfe103f53cbcb039873b1a3e81a9da9abd71995e722318367281d30b35a338bf356662342b653eff38e85881863b7128ddbb856d8ae158365550bb6330b93d4ef0".HexToByteArray(),
                D = "0153603164bcef5c9f62388d06dcbf5681479be4397c07ff6f44bb848465e3397537d5f61abc7bc9266d4df6bae1df4847fcfd3dabdda37a2fe549b821ea858d088d".HexToByteArray(),
            };

        internal static readonly ECDsaCngKeyValues ECDsabrainpoolP160r1_PublicKey =
            new ECDsaCngKeyValues()
            {
                QX = "424D9D7A9F07A0D25C6C26A589EE5020C62C367A".HexToByteArray(),
                QY = "61B4EA3D5940A0AEBCC41B7C1FAF1CF7F477005D".HexToByteArray(),
            };

        internal static readonly byte[] ECDsa521Certificate =
             ("308201a93082010ca00302010202102c3134fe79bb9daa48df6431f4c1e4f3300906072a8648ce3d04013014311230100603"
            + "5504031309456335323155736572301e170d3135303931303231333832305a170d3136303931303033333832305a30143112"
            + "30100603550403130945633532315573657230819b301006072a8648ce3d020106052b8104002303818600040134af29d1fe"
            + "5e581fd2ff6194263abcb6f8cb4d9c08bdb384ede9b8663ae2f4e1af6c85eacc69dc768fbfcd856630792e05484cefb1fefb"
            + "693081dc6490dac579c000bfe103f53cbcb039873b1a3e81a9da9abd71995e722318367281d30b35a338bf356662342b653e"
            + "ff38e85881863b7128ddbb856d8ae158365550bb6330b93d4ef0300906072a8648ce3d040103818b0030818702420090bdf5"
            + "dfb328501910da4b02ba3ccd41f2bb073608c55f0f2b2e1198496c59b44db9e516a6a63ba7841d22cf590e39d3f09636d0eb"
            + "cd59a92c105f499e1329615602414285111634719b9bbd10eb7d08655b2fa7d7eb5e225bfdafef15562ae2f9f0c6a943a7bd"
            + "f0e39223d807b5e2e617a8e424294d90869567326531bcad0f893a0f3a").HexToByteArray();

        internal static readonly byte[] EccCert_KeyAgreement = (
            "308201553081FDA00302010202105A1C956450FFED894E85DC61E11CD968300A" +
            "06082A8648CE3D04030230143112301006035504030C09454344482054657374" +
            "301E170D3135303433303138303131325A170D3136303433303138323131325A" +
            "30143112301006035504030C094543444820546573743059301306072A8648CE" +
            "3D020106082A8648CE3D0301070342000477DE73EA00A82250B69E3F24A14CDD" +
            "C4C47C83993056DD0A2C6C17D5C8E7A054216B9253533D12C082E0C8B91B3B10" +
            "CDAB564820D417E6D056E4E34BCCA87301A331302F300E0603551D0F0101FF04" +
            "0403020009301D0603551D0E0416041472DE05F588BF2741C8A28FF99EA399F7" +
            "AAB2C1B3300A06082A8648CE3D040302034700304402203CDF0CC71C63747BDA" +
            "2D2D563115AE68D34867E74BCA02738086C316B846CDF2022079F3990E5DCCEE" +
            "627B2E6E42317D4D279181EE695EE239D0C8516DD53A896EC3").HexToByteArray();

        internal static readonly byte[] ECDsa224Certificate = (
            "3082026630820214A003020102020900B94BCCE3179BAA21300A06082A8648CE" +
            "3D040302308198310B30090603550406130255533113301106035504080C0A57" +
            "617368696E67746F6E3110300E06035504070C075265646D6F6E64311E301C06" +
            "0355040A0C154D6963726F736F667420436F72706F726174696F6E3120301E06" +
            "0355040B0C172E4E4554204672616D65776F726B2028436F7265465829312030" +
            "1E06035504030C174E4953542F53454320502D3232342054657374204B657930" +
            "1E170D3135313233313232353532345A170D3136303133303232353532345A30" +
            "8198310B30090603550406130255533113301106035504080C0A57617368696E" +
            "67746F6E3110300E06035504070C075265646D6F6E64311E301C060355040A0C" +
            "154D6963726F736F667420436F72706F726174696F6E3120301E060355040B0C" +
            "172E4E4554204672616D65776F726B2028436F72654658293120301E06035504" +
            "030C174E4953542F53454320502D3232342054657374204B6579304E30100607" +
            "2A8648CE3D020106052B81040021033A000452FF02B55AE35AA7FFF1B0A82DC2" +
            "260083DD7D5893E85FBAD1D663B718176F7D5D9A04B8AEA968E9FECFEE348CDB" +
            "49A938401783BADAC484A350304E301D0603551D0E041604140EA9C5C4681A6E" +
            "48CE64E47EE8BBB0BA5FF8AB3E301F0603551D230418301680140EA9C5C4681A" +
            "6E48CE64E47EE8BBB0BA5FF8AB3E300C0603551D13040530030101FF300A0608" +
            "2A8648CE3D040302034000303D021D00AC10B79B6FD6BEE113573A1B68A3B771" +
            "3B9DA2719A9588376E334811021C1AAC3CA829DA79CE223FA83283E6F0A5A59D" +
            "2399E140D957C1C9DDAF").HexToByteArray();

        internal static readonly byte[] ECDsaP256_DigitalSignature_Pfx_Windows = (
            "308204470201033082040306092A864886F70D010701A08203F4048203F03082" +
            "03EC3082016D06092A864886F70D010701A082015E0482015A30820156308201" +
            "52060B2A864886F70D010C0A0102A081CC3081C9301C060A2A864886F70D010C" +
            "0103300E0408EC154269C5878209020207D00481A80BAA4AF8660E6FAB7B050B" +
            "8EF604CFC378652B54FE005DC3C7E2F12E5EFC7FE2BB0E1B3828CAFE752FD64C" +
            "7CA04AF9FBC5A1F36E30D7D299C52BF6AE65B54B9240CC37C04E7E06330C24E9" +
            "6D19A67B7015A6BF52C172FFEA719B930DBE310EEBC756BDFF2DF2846EE973A6" +
            "6C63F4E9130083D64487B35C1941E98B02B6D5A92972293742383C62CCAFB996" +
            "EAD71A1DF5D0380EFFF25BA60B233A39210FD7D55A9B95CD8A440DF666317430" +
            "1306092A864886F70D0109153106040401000000305D06092B06010401823711" +
            "0131501E4E004D006900630072006F0073006F0066007400200053006F006600" +
            "7400770061007200650020004B00650079002000530074006F00720061006700" +
            "65002000500072006F007600690064006500723082027706092A864886F70D01" +
            "0706A0820268308202640201003082025D06092A864886F70D010701301C060A" +
            "2A864886F70D010C0106300E0408175CCB1790C48584020207D080820230E956" +
            "E38768A035D8EA911283A63F2E5B6E5B73231CFC4FFD386481DE24B7BB1B0995" +
            "D614A0D1BD086215CE0054E01EF9CF91B7D80A4ACB6B596F1DFD6CBCA71476F6" +
            "10C0D6DD24A301E4B79BA6993F15D34A8ADB7115A8605E797A2C6826A4379B65" +
            "90B56CA29F7C36997119257A827C3CA0EC7F8F819536208C650E324C8F884794" +
            "78705F833155463A4EFC02B5D5E2608B83F3CAF6C9BB97C1BBBFC6C5584BDCD3" +
            "9C46A3944915B3845C41429C7792EB4FA3A7EDECCD801F31A4B6EF57D808AEEA" +
            "AF3D1F55F378EF8EF9632CED16EDA3EFBE4A9D5C5F608CA90A9AC8D3F86462AC" +
            "219BFFD0B8A87DDD22CF029230369B33FC2B488B5F82702EFC3F270F912EAD2E" +
            "2402D99F8324164C5CD5959F22DEC0D1D212345B4B3F62848E7D9CFCE2224B61" +
            "976C107E1B218B4B7614FF65BCCA388F85D6920270D4C588DEED323C416D014F" +
            "5F648CC2EE941855EB3C889DCB9A345ED11CAE94041A86ED23E5789137A3DE22" +
            "5F4023D260BB686901F2149B5D7E37102FFF5282995892BDC2EAB48BD5DA155F" +
            "72B1BD05EE3EDD32160AC852E5B47CA9AEACE24946062E9D7DCDA642F945C9E7" +
            "C98640DFAC7A2B88E76A560A0B4156611F9BE8B3613C71870F035062BD4E3D9F" +
            "D896CF373CBFBFD31410972CDE50739FFB8EC9180A52D7F5415EBC997E5A4221" +
            "349B4BB7D53614630EEEA729A74E0C0D20726FDE5814321D6C265A7DC6BA24CA" +
            "F2FCE8C8C162733D58E02E08921E70EF838B95C96A5818489782563AE8A2A85F" +
            "64A95EB350FF8EF6D625AD031BCD303B301F300706052B0E03021A0414C8D96C" +
            "ED140F5CA3CB92BEFCA32C690804576ABF0414B59D4FECA9944D40EEFDE7FB96" +
            "196D167B0FA511020207D0").HexToByteArray();

        // The PFX in ECDsaP256_DigitalSignature_Pfx_Windows washed through OpenSSL
        internal static readonly byte[] ECDsaP256_DigitalSignature_Pfx_OpenSsl = (
            "308203BE0201033082038406092A864886F70D010701A0820375048203713082" +
            "036D308201FF06092A864886F70D010706A08201F0308201EC020100308201E5" +
            "06092A864886F70D010701301C060A2A864886F70D010C0106300E040888F579" +
            "00302DB63A02020800808201B8F5EDB44F8B2572E85E52946B233A47F03DF776" +
            "BC3A05CB74B4145A9D3AE3C7FD61B330194E1E154B89929F3FA3908FEE95512A" +
            "052FDDE8E1913E2CCFD803EE6D868696D86934DCF5300DC951F36BE93E3F4AA2" +
            "096B926CF8410AF77FFA087213F84F17EB1D36B61AF4AAD87288301569239B9A" +
            "B66392ADA3D468DC33F42FCEC3BEE78148CA72686BB733DB89FC951AE92FD0F7" +
            "D5937DE78B1AF984BD13E5127F73A91D40097976AEF00157DCC34B16C1724E5B" +
            "88090A1A2DA7337C72720A7ED8F1A89C09AB4143C3F6D80B1925AB8F744069F6" +
            "399D997827F7D0931DCB5E3B09783D1D8555910906B33AD03759D292021C21A2" +
            "9EA2F29CF9BA4D66E4E69AA9FDCCCB4D49A806DBB804EBEBAED7AE0DD4AD2133" +
            "1482A3CC5DB246CE59998824B7E46F337F8887D990FA1756D6A039D293B243BB" +
            "DCFB19AD613A42C5778E7094EA43C3136EF359209790462A36CF87D89B6D76CF" +
            "BD8C34B8C41D96C83683751B8B067F42017A37D05B599B82B70830B5A93499A0" +
            "A4791F5DAB2143C8DF35EC7E88B71A0990E7F6FEA304CE594C9280D7B9120816" +
            "45C87112B1ED85124533792ABEF8B4946F811FB9FE922F6F786E5BFD7D7C43F6" +
            "48AB43C43F3082016606092A864886F70D010701A0820157048201533082014F" +
            "3082014B060B2A864886F70D010C0A0102A081B43081B1301C060A2A864886F7" +
            "0D010C0103300E0408F58B95D6E307213C02020800048190E0FB35890FFB6F30" +
            "7DD0BD8B10EB10488EAB18702E5AC9F67C557409DF8E3F382D06060FB3B5A08D" +
            "1EA31313E80A0488B4034C8906BD873A5308E412783684A35DBD9EEACF5D090D" +
            "AE7390E3309D016C41133946A6CF70E32BE8002CD4F06A90F5BBCE6BF932EC71" +
            "F634312D315310CE2015B30C51FCC54B60FB3D6E7B734C1ADEBE37056A46AB3C" +
            "23276B16603FC50C318184302306092A864886F70D01091531160414F20D17B7" +
            "9B898999F0AA1D5EA333FAEF2BDB2A29305D06092B060104018237110131501E" +
            "4E004D006900630072006F0073006F0066007400200053006F00660074007700" +
            "61007200650020004B00650079002000530074006F0072006100670065002000" +
            "500072006F0076006900640065007230313021300906052B0E03021A05000414" +
            "96C2244022AB2B809E0F97270F7F4EA7769DD26F04084C0E2946D65F8F220202" +
            "0800").HexToByteArray();

        internal struct ECDsaCngKeyValues
        {
            public byte[] QX;
            public byte[] QY;
            public byte[] D;
        }

        internal static readonly RSAParameters RsaBigExponentParams = new RSAParameters
        {
            Modulus = (
                "AF81C1CBD8203F624A539ED6608175372393A2837D4890E48A19DED369731156" +
                "20968D6BE0D3DAA38AA777BE02EE0B6B93B724E8DCC12B632B4FA80BBC925BCE" +
                "624F4CA7CC606306B39403E28C932D24DD546FFE4EF6A37F10770B2215EA8CBB" +
                "5BF427E8C4D89B79EB338375100C5F83E55DE9B4466DDFBEEE42539AEF33EF18" +
                "7B7760C3B1A1B2103C2D8144564A0C1039A09C85CF6B5974EB516FC8D6623C94" +
                "AE3A5A0BB3B4C792957D432391566CF3E2A52AFB0C142B9E0681B8972671AF2B" +
                "82DD390A39B939CF719568687E4990A63050CA7768DCD6B378842F18FDB1F6D9" +
                "FF096BAF7BEB98DCF930D66FCFD503F58D41BFF46212E24E3AFC45EA42BD8847").HexToByteArray(),

            Exponent = new byte[] { 0x02, 0x00, 0x00, 0x04, 0x41 },

            D = (
                "64AF9BA5262483DA92B53F13439FD0EF13012F879ABC03CB7C06F1209904F352" +
                "C1F223519DC48BFAEEBB511B0D955F6167B50E034FEA2ABC590B4EA9FBF0C51F" +
                "9FFEA16F7927AE681CBF7358452BCA29D58705E0CAA106013B09A6F5F5911498" +
                "D2C4FD6915585488E5F3AD89836C93C8775AFAB4D13C2014266BE8EE6B8AA66C" +
                "9E942D493466C8E3A370F8E6378CE95D637E03673670BE4BCACE5FCDADD238D9" +
                "F32CA35DE845776AC4BF36118812328C493F91C25A9BD42672D0AFAFDE0AF7E6" +
                "19078D48B485EF91933DDCFFB54587B8F512D223C81894E91784982F3C5C6587" +
                "1351F4655AB023C4AD99B6B03A96F9046CE124A471E828F05F8DB3BC7CCCF2D1").HexToByteArray(),

            P = (
                "E43A3826A97204AE3CD8649A84DB4BBF0725C4B08F8C43840557A0CD04E313AF" +
                "6D0460DDE69CDC508AD043D72514DA7A66BC918CD9624F485644B9DEEAB2BE0E" +
                "112956D472CF0FD51F80FD33872D2DCC562A0588B012E8C90CE7D254B94792C6" +
                "E7A02B3CCAA150E67A64377ACC49479AD5EB555493B2100CB0410956F7D73BF5").HexToByteArray(),

            Q = (
                "C4DD2D7ADD6CA50740D3973F40C4DEBDBAB51F7F5181ABAE726C32596A3EDD0A" +
                "EE44DAADDD8A9B7A864C4FFDAE00C4CB1F10177BA01C0466F812D522610F8C45" +
                "43F1C3EF579FA9E13AE8DA1A4A8DAE307861D2CEAC03560279B61B6514989883" +
                "FE86C5C7420D312838FC2F70BED59B5229654201882664CEFA38B48A3723E9CB").HexToByteArray(),

            DP = (
                "09ECF151F5CDD2C9E6E52682364FA5B4ED094F622E4031BF46B851358A584DCC" +
                "B5328B0BD9B63589183F491593D2A3ACAD14E0AACDA1F181B5C7D93C57ED26E6" +
                "2C9FC26AF37E4A0644ECE82A7BA8AED88FF1D8E9C56CC66385CDB244EB3D57D1" +
                "7E6AD420B19C9E2BEE18192B816265B74DA55FA3825F922D9D8E835B76BF3071").HexToByteArray(),

            DQ = (
                "89B33B695789174B88368C494639D4D3267224572A40B2FE61910384228E3DBD" +
                "11EED9040CD03977E9E0D7FC8BFC4BF4A93283529FF1D96590B18F4EABEF0303" +
                "794F293E88DC761B3E23AFECB19F29F8A4D2A9058B714CF3F4D10733F13EA72B" +
                "BF1FBEC8D71E106D0CE2115F3AD2DE020325C3879A091C413CD6397F83B3CB89").HexToByteArray(),

            InverseQ = (
                "7C57ED74C9176FBA76C23183202515062C664D4D49FF3E037047A309DA10F159" +
                "0CE01B7A1CD1A4326DC75883DFF93110AB065AAED140C9B98176A8810809ADEC" +
                "75E86764A0951597EF467FA8FD509181CD2E491E43BE41084E5BE1B562EE76E9" +
                "F92C9AB1E5AEAD9D291A6337E4DE85BDE67A0D72B4E55ADCF207F7A5A5225E15").HexToByteArray()
        };

        internal static readonly byte[] BigExponentPkcs10Bytes = (
            "30820311308201F902010030818A310B30090603550406130255533113301106" +
            "03550408130A57617368696E67746F6E3110300E060355040713075265646D6F" +
            "6E64311E301C060355040A13154D6963726F736F667420436F72706F72617469" +
            "6F6E3120301E060355040B13172E4E4554204672616D65776F726B2028436F72" +
            "6546582931123010060355040313096C6F63616C686F737430820124300D0609" +
            "2A864886F70D010101050003820111003082010C0282010100AF81C1CBD8203F" +
            "624A539ED6608175372393A2837D4890E48A19DED36973115620968D6BE0D3DA" +
            "A38AA777BE02EE0B6B93B724E8DCC12B632B4FA80BBC925BCE624F4CA7CC6063" +
            "06B39403E28C932D24DD546FFE4EF6A37F10770B2215EA8CBB5BF427E8C4D89B" +
            "79EB338375100C5F83E55DE9B4466DDFBEEE42539AEF33EF187B7760C3B1A1B2" +
            "103C2D8144564A0C1039A09C85CF6B5974EB516FC8D6623C94AE3A5A0BB3B4C7" +
            "92957D432391566CF3E2A52AFB0C142B9E0681B8972671AF2B82DD390A39B939" +
            "CF719568687E4990A63050CA7768DCD6B378842F18FDB1F6D9FF096BAF7BEB98" +
            "DCF930D66FCFD503F58D41BFF46212E24E3AFC45EA42BD884702050200000441" +
            "A03F303D06092A864886F70D01090E3130302E302C0603551D11042530238704" +
            "7F00000187100000000000000000000000000000000182096C6F63616C686F73" +
            "74300D06092A864886F70D01010B050003820101003BCAE7E02D3A828435110C" +
            "8A65197FF1A027EC5ACA37EBE29B6E7093A4BDCA9BDA8E244DC05846AA9F186D" +
            "2EBBDF6474BB09ECF5A3C11F3A7E56D9D489C3D4AE2DCF5D52ABFCDFED6D4623" +
            "AF7C7D2E52A189BC4A0BFC5EB96EC158A96B292DF6E4ADCAE5233A7E1598444E" +
            "23F732526B71172266E45706F90EFAB0945A75D446F0A6547C788DD81AD6F4D1" +
            "E7FD0E8884083AF52003D9CD38B3A140F2E552CF3FBF0B4C771E5745C6DA6F26" +
            "DCFD0FEB87B9FDD2F4724A09DE1FB4C55E439F43C6E37A866BA19494B210D294" +
            "699B3C957C6DD22E9B63DBAE3B5AE62919F0EA3DF304C7DD9E0BBA0E7053605F" +
            "D066A788426159BB937C58E5A110461DC9364CA7CA").HexToByteArray();

        internal static byte[] T61StringCertificate = (
            "2d2d2d2d2d424547494e2043455254494649434154452d2d2d2d2d0a4d494943" +
            "4544434341626f4343514348477057667a4678366e6a414e42676b71686b6947" +
            "3977304241517346414443426a6a45614d42674741315545417851520a644756" +
            "7a6443317a5a584a325a5849756247396a59577778437a414a42674e56424159" +
            "54416c56544d52597746415944565151494641314e59584e7a59574e6f0a6458" +
            "4e6c6448527a4d524177446759445651514b4641645959573168636d6c754d52" +
            "5177456759445651514c46417446626d6470626d566c636d6c755a7a456a0a4d" +
            "43454743537147534962334451454a4152595562574669595856735147317059" +
            "334a766332396d6443356a623230774868634e4d5467774e7a49314d5467310a" +
            "4e5441785768634e4d6a67774e7a49794d5467314e544178576a43426a6a4561" +
            "4d42674741315545417851526447567a6443317a5a584a325a5849756247396a" +
            "0a59577778437a414a42674e5642415954416c56544d52597746415944565151" +
            "494641314e59584e7a59574e6f64584e6c6448527a4d52417744675944565151" +
            "4b0a4641645959573168636d6c754d525177456759445651514c46417446626d" +
            "6470626d566c636d6c755a7a456a4d43454743537147534962334451454a4152" +
            "59550a62574669595856735147317059334a766332396d6443356a6232307758" +
            "44414e42676b71686b6947397730424151454641414e4c41444249416b454131" +
            "366a590a51424d417459646372394d3534426677684b695852675855566f7031" +
            "4f5a4e735a77714d714872543557674d50386b3565344a63444175417455434f" +
            "6d7761670a52697647463735684a5449375963764b69514944415141424d4130" +
            "4743537147534962334451454243775541413045415377736f42336436506c72" +
            "65433864710a4f38507370457171576a4e69464358724b317169547865584336" +
            "536844674d445733464670316c6d33564747564637365078504a792f4c6c6249" +
            "39366d5a59590a734f65735a513d3d0a2d2d2d2d2d454e442043455254494649" +
            "434154452d2d2d2d2d0a").HexToByteArray();

        internal static byte[] EmptySubjectCertificate = (
            "308202A73082018FA003020102020103300D06092A864886F70D01010B050030" +
            "1F311D301B06035504031314456D707479205375626A65637420497373756572" +
            "301E170D3138303631353134343030365A170D3333303631353134323030365A" +
            "300030820122300D06092A864886F70D01010105000382010F003082010A0282" +
            "0101008BFA2727E0D93BEF2A992E912A829FF7A374992F2BD910291BF2BD747D" +
            "5CCCF997276ABC2F1CACEAD3F964333F5FF9D7F116A0AC392E711866CCEB0E48" +
            "80716367613E4ABD26FCB946E0A2C6AB84ABFD1EF377CD4F6C497D49D9B99CBE" +
            "DA878CA4E962307DE110345D2B22CB95A2CC3E0AB94D505CF43FA3B0343B4957" +
            "AF361E3604507732150254D77162909887509D5990499C039E5C3871326E09C2" +
            "DE132786032B4CD9F460CE35DD0650BE77B0AD9963BC498773CAC858AD15000E" +
            "A97793A906D5052A381AA2EE84BF2734833BF12DE932CF67A6D567E627898D96" +
            "2FEA2B4F55DE992C205DD67B5B2A59CDD25C04070363D48C8ED854BE013C5D1E" +
            "3E57FF0203010001A30D300B30090603551D1304023000300D06092A864886F7" +
            "0D01010B0500038201010001BF6D51DA7C45965CB2B6B889E0B1875DFEFDBAC1" +
            "558978A53E37BB796E10CBA9AFB698168F55DBD2EC4D26E9FABB7D40D55A2FCA" +
            "57F7BBC8509D1B88EB753468F3B57A42718081F00430115A48035CE72CBF9294" +
            "2837DF2C2FDC38CE213F2258C5E071A4E18ED70DFC281CD376E84ADC92922416" +
            "43389C87098AFAD976F811AA95B48B69DAF7CB31C79953BBFD1C96839561CE12" +
            "435F83CEA9F2CB9A94235B0B21FB0D591CDC41676FE927E41CC3A776FAF97146" +
            "BF14B08041CB1A90AC7339E7BD1DAF9A600479754F42C88D418B5449F5D34050" +
            "4E543013489C47297C83440C3EEF49C9A9D96E398949F03ACB0D5F72E7B4E9D3" +
            "895D82CAB526DB74AC5629").HexToByteArray();

        internal static byte[] EmptySubjectIssuerCertificate = (
            "308202D4308201BCA003020102020900F68AD256DB542CF9300D06092A864886" +
            "F70D01010B0500301F311D301B06035504031314456D707479205375626A6563" +
            "7420497373756572301E170D3137303631353134343030365A170D3333303631" +
            "353134343030365A301F311D301B06035504031314456D707479205375626A65" +
            "63742049737375657230820122300D06092A864886F70D01010105000382010F" +
            "003082010A0282010100A67F5898CCA5FC235EEB2FF14BF0BF490BB507C4D552" +
            "76E0D86CAA72BFDBDE7E7F38EC3184B18D32AEA7F5A1EEF0D2D24B7B8ED340F5" +
            "00703D3758B7E3824848CD9A4CDF15F73EBB1D4A02ED8182673138822C148463" +
            "B6126D14BE03C9A4DF62D35109BF7A8CDDFF6AE5A55C75496C13376C9B0096A0" +
            "5F398703FFBB6B69F7EA79B1F1F955F07CEBDAF3FB87D6E6F9C33678C49EC5E5" +
            "7EF10AEB14E009C83DC3DD3A2707F36D1A8723849DCD020CFBB0D38972B15F12" +
            "53209E015915A2275ADFB0164DB5A6C79BA53EA5782B001D92764D21694E5992" +
            "33B4D2C7FE260F925194C372EE473812B4F82381B4027E1F7F52E72A6ECC5BEE" +
            "656FE161E7681A06A9BF0203010001A3133011300F0603551D130101FF040530" +
            "030101FF300D06092A864886F70D01010B050003820101008D239025B8266EFF" +
            "AEB1B76EA159ED7BF6A6F2E5A23266DF07767DEDF26B43983EC8717AC2E6B99E" +
            "A21625D787D39BF94110D5A9B37CF0E6FB3062090D280B889A0E8DA5952E46D5" +
            "107F9BE829119282ADD53414CE0BA47C1E04D82BCC50118BB9CB3905E22FADD5" +
            "1905F23373D7F5E9E32C5C7D27B68CD15D1E079153B66B3DBE148D4DF2BF8EA1" +
            "8E68D838BE1C86E68E4CAF09A90F7428A4A50922BCB01010A1B585227930CEE0" +
            "027D561804C2F023AEBAA63F312F49A37B74BED2DB8B6D8AA1A6E60320589EB5" +
            "B14EB93CC47DAF7F07D67C25FD275493C0AF0CC23E8AED47963DCCE7C2ADFD56" +
            "5B8A77118C2978E15D1511572C4496CD1180094D43AE50FD").HexToByteArray();

        internal static DSAParameters GetDSA1024Params()
        {
            DSAParameters p = new DSAParameters();
            p.G = (
                "6BC366B66355545E098F1FE90E5469B567E09FA79D817F2B367B45DECD4301A59C81D6911F7691D370E15AC692C04BC11872" +
                "C171A7FE654E963D7DDA575A9E98CE026FB7D3934A258608134A8EC5ED69A2AEDC89401B67ADDE427F17EDAEB72D7AF45D9A" +
                "B1D59E1B13D4EFBD17C764330267DDE352C20E05B80DB3C109FE8B9C").HexToByteArray();
            p.P = (
                "C16D26C74D6C1627799C0918548E553FE58C7881DA484629CAF64311F4B27CFEF6BDB0F21206B0FFC4999A2FED53B43B9EE2" +
                "910C68DA2C436A8018F4938F6472369F5647D005BCC96E22590CC15E3CD4EA0D132F5DA5AF6AAA0807B0CC4EF3404AF542F4" +
                "546B37BDD6A47E641130837DB99397C845635D7DC36D0537E4A84B31").HexToByteArray();
            p.Q = "D83C0ECB73551E2FE30D51FCF4236C651883ADD7".HexToByteArray();
            p.X = "C02678007779E52E360682214BD47F8FAF42BC2D".HexToByteArray();
            p.Y = (
                "690BB37A9145E05D6E7B47C457898AAEDD72501C9D16E79B1AD75A872CF017AA90BBFB90F1B3B7F5C03C87E46E8725665526" +
                "FD34157B26F596A1F0997F59F3E65EFC615A552D5E7569C5FFC4593D5A0299110E71C97E1221A5A03FE9A6935AEDD88EF0B3" +
                "B2F79D3A99ED75F7B871E6EAF2680D96D574A5F4C13BACE3B4B44DE1").HexToByteArray();
            return p;
        }
    }
}
